
#include "VarioSynthesiser.hpp"

#ifdef TESTBENCH
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>



void wait_ms (unsigned ms) { usleep(ms*1000); }

VarioSynthesiser::VarioSynthesiser(unsigned tpl)
{
//  bool enabled = true;
  override_vario = 0;
  time_slice = 0;
  vario_freq = 0;
  active_freq = 0;
  vario_timeticks = 0;
  ticks_per_loop = tpl;
  SetDefaults();
  SetSilence();
}
#else
VarioSynthesiser::VarioSynthesiser(unsigned tpl, AnalogOut * s, Serial * ch)
{
//  bool enabled = true;
  vSound = s;
  cChan = ch;
  override_vario = 0;
  ack_tone_active = false;
  time_slice = 0;
  vario_freq = 0;
  active_freq = 0;
  vario_timeticks = 0;
  ticks_per_loop = tpl;
  SetDefaults();
  SetSilence();
}
#endif

int Clamp(const int value, const int min, const int max)
{
  return (value < min)
    ? min
    : ((value > max)
       ? max : value);
}


/**
 * The minimum and maximum vario range for the constants below [cm/s].
 */
static int min_vario = -500, max_vario = 500;

void
VarioSynthesiser::timerInit(int hz) {
     __TIM6_CLK_ENABLE();


    // Time base clock configuration
    TIM6_Handle.Instance = TIM6;
    TIM6_Handle.Init.ClockDivision = TIM_CLOCKDIVISION_DIV1; // unused for TIM6
    TIM6_Handle.Init.CounterMode = TIM_COUNTERMODE_UP; // unused for TIM6
    SetTone(hz);

    // TIM6 TRGO selection
    TIM_MasterConfigTypeDef config;
    config.MasterOutputTrigger = TIM_TRGO_UPDATE;
    config.MasterSlaveMode = TIM_MASTERSLAVEMODE_DISABLE;
    HAL_TIMEx_MasterConfigSynchronization(&TIM6_Handle, &config);

    // TIM6 start counter
    HAL_TIM_Base_Start(&TIM6_Handle);
}

void
VarioSynthesiser::dacInit() {
    hdac.Instance = DAC;
    hdac.State = HAL_DAC_STATE_RESET;
    HAL_DAC_Init(&hdac);

    DAC_ChannelConfTypeDef config;
    config.DAC_Trigger = DAC_TRIGGER_T6_TRGO;
    config.DAC_OutputBuffer = DAC_OUTPUTBUFFER_ENABLE;
    HAL_DAC_ConfigChannel(&hdac, &config, DAC_CHANNEL_1);
    HAL_DAC_Start_DMA(&hdac,DAC_CHANNEL_1,(uint32_t *)sound_data,NSAMP, DAC_ALIGN_12B_R);
}

void
VarioSynthesiser::dmaInit() {

    __DMA1_CLK_ENABLE();
    DMA_Handle.Instance = DMA1_Stream5;

    // Need to deinit DMA first
    DMA_Handle.State = HAL_DMA_STATE_READY;
    HAL_DMA_DeInit(&DMA_Handle);

    DMA_Handle.Init.Channel = DMA_CHANNEL_7;
    DMA_Handle.Init.Direction = DMA_MEMORY_TO_PERIPH;
    DMA_Handle.Init.PeriphInc = DMA_PINC_DISABLE;
    DMA_Handle.Init.MemInc = DMA_MINC_ENABLE;
    DMA_Handle.Init.PeriphDataAlignment = DMA_PDATAALIGN_HALFWORD;
    DMA_Handle.Init.MemDataAlignment = DMA_MDATAALIGN_HALFWORD;
    DMA_Handle.Init.Mode = DMA_CIRCULAR;  // NORMAL or CIRCULAR
    DMA_Handle.Init.Priority = DMA_PRIORITY_HIGH;
    DMA_Handle.Init.FIFOMode = DMA_FIFOMODE_DISABLE;
    DMA_Handle.Init.FIFOThreshold = DMA_FIFO_THRESHOLD_HALFFULL;
    DMA_Handle.Init.MemBurst = DMA_MBURST_SINGLE;
    DMA_Handle.Init.PeriphBurst = DMA_PBURST_SINGLE;
    HAL_DMA_Init(&DMA_Handle);

    __HAL_LINKDMA(&hdac, DMA_Handle1, DMA_Handle);
}

  
void
VarioSynthesiser::SetDefaults()
{
//  bool enabled = true;
  ivolume = 10;
  dead_band_enabled = true;
  dualtone = true;

  min_frequency = 120;
  zero_frequency = 500;
  max_frequency = 1500;

  zero_timeticks = 1;
  max_timeticks = ticks_per_loop - 1;

  min_dead = -0.3;
  max_dead = 0.1;
  
#ifndef TESTBENCH
  uint16_t i;
  /*
  // Sine wave
  float phase = 3.14159 * 2./NSAMP;
  for (i=0;i<NSAMP;i++) master_sound_data[i] = (int16_t)(2040.0 * sinf(i*phase));
  */

  /*
  // Square wave
  for (i=0;i<(NSAMP/2);i++) master_sound_data[i]= 2040;
  for (;i<NSAMP;i++) master_sound_data[i]= -2040;
  */

  // Sawtooth
  int16_t step= 4 * 2040 / NSAMP;
  int16_t v = 0;
  for (i=0;i<(NSAMP/4);i++) {master_sound_data[i]=v; v+=step;}
  for (;i<(3*NSAMP/4);i++) {master_sound_data[i]=v; v-=step;}
  for (;i<NSAMP;i++) {master_sound_data[i]=v; v+=step;}

  // for (i=0;i<NSAMP;i++) cChan->printf("%d:%d\n",i,master_sound_data[i]);
  UpDownVolume(0);
  // cChan->printf("\nFrequency %d Hz\n",zero_frequency);
  timerInit(zero_frequency);
  dmaInit();
  dacInit();

#endif

  SetTone(min_frequency); wait_ms(100);
  SetTone(max_frequency); wait_ms(100);
  SetTone(zero_frequency); wait_ms(100);
  SetSilence();
}

int
VarioSynthesiser::UpDownVolume(int incr)
{
  ivolume = Clamp(ivolume + incr,0,100);
  for (uint16_t i=0;i<NSAMP;i++) { v_sound_data[i] = 2047 + (ivolume * master_sound_data[i] / 100); }

  /*****
  int avg = 0;
  for (uint16_t i=0;i<NSAMP;i++) { avg += v_sound_data[i]; }
  avg = avg / NSAMP;
  cChan->printf("V: %d, F: %d, M: %d\n",ivolume,active_freq,avg);
  ***/
#ifdef TESTBENCH
  printf ("Volume: %3.1f\n",ivolume/100.0f);
#endif
  return (ivolume);
}



unsigned
VarioSynthesiser::VarioToFrequency(int ivario)
{
  return ivario > 0
    ? (zero_frequency + (unsigned)ivario * (max_frequency - zero_frequency) / (unsigned)max_vario)
    : (zero_frequency - (unsigned)(ivario * (int)(zero_frequency - min_frequency) / min_vario));
}

unsigned
VarioSynthesiser::VarioToTimeTicks(int ivario)
{
  return ivario > 0
    ? (zero_timeticks + (unsigned)ivario * (max_timeticks - zero_timeticks) / (unsigned)max_vario)
    : (1000);
}

void 
VarioSynthesiser::SetTone(unsigned freq)
{

  if (abs((int)freq-(int)active_freq) < 10) {return;}
  // cChan->printf("f %d -> %d Hz\n",active_freq,freq);

  active_freq = freq;
#ifndef TESTBENCH
    int hz = active_freq * NSAMP;

    // Compute the prescaler value so TIM6 triggers at freq-Hz
    uint32_t period = 2*HAL_RCC_GetPCLK1Freq()/hz;  // ? 2 times PCLK1 see pwm_api
    uint32_t prescaler = 1;
    while (period > 0xfff) {
        period >>= 1;
        prescaler <<= 1;
    }
    // cChan->printf("f: %d Hz, period: %d, prescaler: %d\n",active_freq,period,prescaler);

    // Time base clock configuration
    // TIM6_Handle.Instance = TIM6;
    TIM6_Handle.Init.Period = period - 1;
    TIM6_Handle.Init.Prescaler = prescaler - 1;
    // TIM6_Handle.Init.ClockDivision = TIM_CLOCKDIVISION_DIV1; // unused for TIM6
    // TIM6_Handle.Init.CounterMode = TIM_COUNTERMODE_UP; // unused for TIM6
    HAL_TIM_Base_Init(&TIM6_Handle);

  // vModulation->write(VSOUND_ON);
  SetSound();
#endif
}

void
VarioSynthesiser::SetVario(float vario)
{
  const int ivario = Clamp((int)(vario * 100), min_vario, max_vario);
  time_slice = 0;
  vario_freq = VarioToFrequency(ivario);
  vario_timeticks = VarioToTimeTicks(ivario);
  dual_tone_base_ticks = vario_timeticks / 2;
  in_dead_band = InDeadBand(ivario) && dead_band_enabled;

  if (in_dead_band)
    {
    SetSilence();
    return;
    }

  if (!ack_tone_active) {
    if ((ivario > 0) && dualtone && (dual_tone_base_ticks > 0)) {
	SetTone(zero_frequency);
	} else {
	SetTone(vario_freq);
	}
      }
  return;
}

void
VarioSynthesiser::TimeTick()
{
  time_slice += 1;

  if (override_vario > 0) {
    SetTone(ack_freq);
    override_vario -= 1;
    return;
    }

  if (ack_tone_active) {
    ack_tone_active = false;
    SetSilence();
    return; }

  if (in_dead_band || (time_slice >= vario_timeticks)) {
    SetSilence();
    return; }

  if (time_slice >= dual_tone_base_ticks) SetTone(vario_freq);
}

void
VarioSynthesiser::AckTone(unsigned freq,unsigned ticks)
{
  if (ticks > 0) {
    override_vario = ticks-1;
    ack_freq = freq;
    SetTone(ack_freq);
    ack_tone_active = true;
  }
}

void
VarioSynthesiser::SetSilence()
{
  active_freq = 0;
  for (uint16_t i=0;i<NSAMP;i++) { sound_data[i] = 2047; }
}

void
VarioSynthesiser::SetSound()
{
  for (uint16_t i=0;i<NSAMP;i++) { sound_data[i] = v_sound_data[i]; }
}


