/**
* @file main.cpp
* @brief This file programs the processor for the inductive force sensor
* using the library LDC1101.h and LDC1101.cpp.
* - Led: processing communication with LDC1101
*
* Log protocol:
* -  1 minute  at 20 Hz
* - 29 minutes rest
*
* @author Bob Giesberts
*
* @date 2015-12-17
*/
#include "mbed.h"
#include "LDC1101.h"        // inductive force sensor
#include "SDFileSystem.h"   // control the SD card
#include "Bob.h"            // processorboard
#include "DS1825.h"         // thermometer

#include <iostream>
#include <vector>
#include <string>
using namespace std;

// SETTINGS
bool DEBUG              = true;
float C                 = 120E-12;      // Capacitor in F
uint16_t INTERVAL_FIRST = 600;          // First sampling time in seconds.  10:00 minutes = 600 sec
uint16_t INTERVAL_OFF   = 870;          // Waiting interval in seconds.     14:30 minutes = 870 sec (14*60+30)
uint16_t INTERVAL_ON    = 30;           // Sampling interval in seconds.     0:30 minutes =  30 sec

// Correct pinnames
// PinName _LEDpin = PTB0;     // For the first sensor (a bright red led)
// PinName _Tpin = PTB1;       // For the first sensor 
PinName _LEDpin = PTB1;     // For all other sensors (a faint green led)
PinName _Tpin = PTB0;       // For all other sensors


// load libraries
Bob bob( _LEDpin, // LED
            PTC3, // sd_enable
            PTE0, // sd_present
            PTC2, // battery
            PTE30 // 3V3_DET        TODO: can we use this to detect if we're charging the battery? If so, how should we respond?
            );
Serial pc(USBTX, USBRX);

// timer variables
uint32_t now = 0, next = 0, prev = 0;
uint8_t t_high = 0;
uint32_t t_lost = 0, lost_prev = 0;
uint8_t lost_high = 0;
uint32_t t_sleep = 0;
uint32_t t;

// file variables
FILE *fp;
string filename = "/sd/data00.txt";
const char *fn;

// temperature variables
uint8_t Tdata[9];
int16_t T_bin;
float T;

// temporal storage for data samples
const uint8_t package_size = 18;    // ±12-13 Hz, so 60 would take about 5 sec.
struct mydata {
    uint32_t    t[package_size];    // time (s)
    uint32_t    L[package_size];    // LHR_DATA
} collected;
uint8_t counter = 0;

// function to write all data to the SD card
void storeit( float T )
{   
    // write data to SD card
    bob.ledon();
    fp = fopen( fn, "a" );      // open file (append)
    for( int i = 0; i < counter; i++ )
       fprintf( fp, "%.2f;%d;%.4f;%.4f\r\n", (float) collected.t[i]/100.0, collected.L[i], bob.battery(), T ); // write to file
    fclose( fp );               // close file
    bob.ledoff();
                 
    //pc.printf( "%d points: %.2f;%d;%.4f;%.4f\r\n", counter, (float) collected.t[0]/100.0, collected.L[0], bob.battery(), T );
                    
    // Reset data
    memset(collected.t, 0, counter);
    memset(collected.L, 0, counter);
    counter = 0;    
}


int main(void)
{
    if(DEBUG)  
    {
        // load libraries to take control over the communication        
        LDC1101 *ldc = new LDC1101( PTC6,   // mosi
                                    PTC7,   // miso
                                    PTC5,   // sck
                                    PTC4,   // cs
                                    C,      // capacitor
                                    16E6    // f_CLKIN
                                    );      // load sensor (LDC1101) library (this takes 0.81 s)
        DS1825 *thermometer = new DS1825( _Tpin );                          // load thermometer (DS1825) library
        
        while(1)
        {
            // wait for new data to be ready
            while( !ldc->is_New_LHR_data() ) { }
            
            pc.printf("%.3f\n", ldc->get_fsensor());
            
            //wait(0.1);
            
           
        }
        
        
    }
    
    
    
    
    
    
    
    
    
    
    // Load SD File system
    // - the Raspberry Pie SD cards give a warning here that might just be ignored:
    //      "Not in idle state after sending CMD8 (not an SD card?)
    //       Didn't get a response from the disk
    //       Set 512-byte block timed out"
    // TODO: buy 'better' SD cards from IAPC / the Stores (Kingston ...)
    bob.wakeup();
    SDFileSystem *sd = new SDFileSystem(PTD6, // mosi
                                        PTD7, // miso
                                        PTD5, // sclk
                                        PTD4, // cs
                                        "sd"
                                        );

    // Create a new data file (data00.txt)
    mkdir("/sd", 0777);
    for(uint8_t i = 0; i < 100; i++)
    {
        filename[8] = i/10 + '0';
        filename[9] = i%10 + '0';
        fp = fopen( filename.c_str() , "r" );
        if( fp == NULL ) // read failed so file does not exist
        {
            fp = fopen( filename.c_str(), "w" ); // create it
            if( fp != NULL )
            {
                fn = filename.c_str();
                fclose( fp );
                break;
            } else {
                bob.ledon();  // error: unable to create new file
            }
        } else { // file already exists
            fclose( fp );
        }
    }
    if( fp == NULL )
    {
        bob.ledon();  // error: file 00 - 99 already exists
        // pc.printf( "/sd/data00.txt t/m /sd/data99.txt already exist!\n" );
    }

    // Unload SD File system
    delete sd;
    sd = NULL;
   
    while(1)
    {
        // Wakeup: SD + LDC1101 on
        bob.wakeup();

        // what time is it now?
        prev = now;                                                     // 0 -> 429 496     --> (4 294,96 s)        (71 min)
        now = (uint32_t) clock();                                       // 0 -> 429 496     --> (4 294,96 s)        (71 min)
        if( now < prev )
            t_high++;                                                   // 0 -> 255         --> (255*4 294,96 s)    (12 days)
        t = now + 429496.7296*t_high + t_lost + 429496.7296*lost_high;  // 0 -> 219 901 952 --> (2 199 019,52 s)    (25 days)
        
        // load libraries to take control over the communication        
        LDC1101 *ldc = new LDC1101( PTC6,   // mosi
                                    PTC7,   // miso
                                    PTC5,   // sck
                                    PTC4,   // cs
                                    C,      // capacitor
                                    16E6    // f_CLKIN
                                    );      // load sensor (LDC1101) library (this takes 0.81 s)
        DS1825 *thermometer = new DS1825( _Tpin );                          // load thermometer (DS1825) library
        SDFileSystem *sd = new SDFileSystem(PTD6, // mosi
                                            PTD7, // miso
                                            PTD5, // sclk
                                            PTD4, // cs
                                            "sd"
                                            );  // load SD system on
            mkdir("/sd", 0777); // select folder


        // How long should we takes samples?
        if( next == 0 )
        {
            next = t + INTERVAL_FIRST*100;
        } else {
            next = t + INTERVAL_ON*100;                                 // 0 -> 219 904 952 --> (2 199 049,52 s)    (25 days)
        }

        // Take samples for INTERVAL_ON seconds
        while( t < next )
        {
            // wait for new data to be ready
            while( !ldc->is_New_LHR_data() ) { }

            
            pc.printf("%d\n", ldc->get_LHR_Data());



            // what time is it now?
            prev = now;
            now = (uint32_t) clock();
            if( now < prev ) 
                t_high++;
            t = now + 429496.7296*t_high + t_lost + 429496.7296*lost_high;

            if( !ldc->is_Oscillation_Error() ) // sensor overloaded, this happends when the target is too close to the coil
            {
                // Store data in temporal memory
                collected.t[counter] = t;
                collected.L[counter] = ldc->get_LHR_Data();
                counter++;
            }
   
            // Write a full package of data points to the SD card
            if( counter >= package_size-1  )
                storeit( thermometer->getTemperature() );
        }
        
        // Write remaining data to the SD card
        if( counter > 0 )
            storeit( thermometer->getTemperature() );

        // prepare for sleep, power down the SD and the LDC1101
        delete sd;          sd  = NULL;             // unload library to be able to power down completely
        delete ldc;         ldc = NULL;             // unload library to be able to power down completely
        delete thermometer; thermometer = NULL;     // unload library to be able to power down completely
        bob.beforesleep();
        
        // if the battery is almost empty (Vbatt < 3.10 V), the termometer stops 
        // working well and L can not be corrected. So just shut down...
        if( bob.battery() < 3.10f )
        {
            bob.ledon();  // error: battery empty
            // pc.printf( "Battery empty (%.1f V), shutting down.\n", bob.battery() );
            exit(0);
        }

        // what time is it now?
        prev = now;
        now = (uint32_t) clock();
        if( now < prev ) 
            t_high++;
        t = now + 429496.7296*t_high + t_lost + 429496.7296*lost_high;

        // Calculate sleeping time (correction to INTERVAL_OFF)
        // t has passed the limit of next. Those few ms are substracted from INTERVAL_OFF.
        t_sleep = INTERVAL_OFF*100 - (t - next);

        // Calculate time that will be lost during sleep
        lost_prev = t_lost;
        t_lost += t_sleep;
        if( t_lost < lost_prev ) 
            lost_high++;

        // Sleep now (ms), enter low power mode
        bob.sleep( t_sleep * 10 );

    }




}

