
#include "laser.h"

#define LASER_BAUD_RATE 115200

extern DigitalOut disableLRF;
extern DigitalOut nReset;

uint32_t base = 0x40002000;
uint32_t rxOffset = 0x514;
uint32_t txOffset = 0x50c;

uint32_t rx = base + rxOffset;
uint32_t tx = base + txOffset;

uint32_t* prx = (uint32_t*)rx;
uint32_t* ptx = (uint32_t*)tx;

Laser::Laser(Serial& serial, int n) : timerRunning(false), idx(0), serial(serial), powerOffState (true), busy(false), nSamples(n)
{
}

void Laser::discardResponse()
{
    // char c = 0;
    wait_ms(30);        // wait for the response
    while(serial.readable()) {
        serial.getc();
    }
}

void Laser::triggerDistanceMeasurement()
{
    if(!busy){
        const int bufSize = 30;
        char cmd[bufSize];
        snprintf(&cmd[0], bufSize, "0 -1 %d doMeasDistExt\n", nSamples);
        //char cmd[] = "0 -1 10 doMeasDistExt\n";   // single reading averaged over 10 measurements
        //char cmd[] = "0 -1 100 doMeasDistExt\n";   // single reading averaged over 100 measurements
        //char cmd[] = "0 -1 -1 doMeasDistExt\n";     // single reading auto choice of number of averages - This could make the laser to lock up and may need reseting
    
        sendCommand(cmd);
        processResponse();
        setRedDot(true);
    }
}

bool Laser::sendCommand(char cmd[])
{
    // start timer before the first of the command is sent to the laser
    timer.reset();
    timer.start();
    timerRunning = true;

    for(int i = 0; i < strlen(cmd); i++) {
        serial.putc(cmd[i]);
    }

    return true;
}

bool Laser::processResponse()
{
#define TIMEOUT     25000  // in units of 100us
    int i = 0;
    char c = 0;
    uint16_t count = 0;
    busy = true;
    
    do {
        if(serial.readable()) {
            // stop timer as soon as we have received the first byte of the response. We need to subtract the time of this receiption
            if(timerRunning) {
                timer.stop();
                timerRunning = false;
            }
            c = serial.getc();
            buf[i++] = c;
        } else {
            wait_us(100);
        }
    } while (c != '\n' && i < bufSize && count++ < TIMEOUT);  // timeout after about 500ms and ensure no overflow



    if (count >= TIMEOUT || i >= bufSize) { // timeout or overflow
        // need to reset the LRF module before talking to it again, otherwise it may lock up
         nReset = 0;
        wait_ms(100);
        nReset = 1;
        wait_ms(1000);
        enableMeasurement(true);
        busy = false;
    } else {
        buf[i -1] = 0;
    }

    float distance = -5;
    vector<char*> v;

    split(buf, ' ', v);

    if (v.size() != 6 || atoi(v[1]) != 0 || strcmp(v[5], "Reply") != 0) {
        // there is an error
        distanceCallback(-1.0, 0.0);
    } else {
        float elapsed =  (float)(timer.read_us()/1000.0);      // elapsed in ms
        distance = atoi(v[2]) / 1000000.0; // distance in m
        distanceCallback(distance, elapsed);
    }
    
    busy = false;

    return true;
}

void Laser::enableMeasurement(bool enable)
{
    if (enable) {
        sendCommand(";\n");
        discardResponse();

    } else {
        sendCommand("switchMeasOff\n");
        discardResponse(); 
        sendCommand("2 0 startPwrSave;");
        discardResponse();       
    }
}

void Laser::setRedDot(bool on)
{
    if(on) {
        char cmd[] = "0 3 0 -1 3 doLaserCmd;\n";
        sendCommand(cmd);
        wait_ms(120);       // take a while for a response for to measurement command
        discardResponse();
    } else {
        sendCommand("switchMeasOff\n");
        discardResponse();
    }
}

void Laser::split(char str[], char c, std::vector<char*>& v)
{
    char * pch;
    char limiter[] = {c};
    pch = strtok (str, limiter);
    while (pch != NULL) {
        v.push_back(pch);
        pch = strtok (NULL, limiter);
    }
}

void Laser::setDistaceCallback(void (*callback)(float, float))
{
    distanceCallback = callback;
}

void Laser::setDebugCallback(void (*callback)(char*))
{
    debugCallback = callback;
}

void Laser::turnLaserPowerOn()
{
    if(powerOffState == true) {
        powerOffState = false;

        connectPower();
        enableMeasurement(true);
        setRedDot(true);
    }
}

void Laser::turnLaserPowerOff()
{
    if(powerOffState == false) {
        powerOffState = true;
        setRedDot(false);   // this disables measurements

        removePower();
    }
}

// when connceting power to the laser module ensure that tx and rx pins are
// reassigned to the serial port
void Laser::connectPower()
{
    disableLRF = 0;
    nReset = 1;
    *ptx = 27;  // p27 for tx
    *prx = 26;  // p26 for rx
    wait_ms(1000);
}

// when removing power from the laser module ensure that rx and tx pins are not
// driving voltage into the laser module to avoid hardware damage
void Laser::removePower()
{
    *ptx = 0xffffffff;      // no pin for tx
    *prx = 0xffffffff;      // no pin for rx
    DigitalOut rx(p26);
    DigitalOut tx(p27);
    rx = 0;
    tx = 0;
    nReset = 0;
    disableLRF = 1;
}

Laser::~Laser()
{
    enableMeasurement(false);
}
