#include "Grove_LCD_RGB_Backlight.h"
#include "mbed.h"
#include "DebounceIn.h"


#define OUT_OFF    (0)
#define OUT_ON     (1)

#define IN_OFF     (1)
#define IN_ON      (0)


//input
DebounceIn doorClosed(p5);              //Check door status
DebounceIn cycleStartBtt(p14);          //cycle start button 
DebounceIn stopBtt(p15);                //stop button
//InterruptIn stopBtt(p15);             //stop button
//DigitalIn pot(p20);

//output
DigitalOut doorInterlock(p6);           //Door interlock
DigitalOut fanRelay(p7);                //Fan control
DigitalOut redRelay(p8);                //Red light 
DigitalOut yellowRelay(p11);            //Yellow light 
DigitalOut greenRelay(p12);             //Green light 
DigitalOut damperRelay(p13);            //Damper solenoid control
DigitalOut damperRelayAlt(p16);         //Damper solenoid toggle control                    
DigitalOut vfd(p17);                    //VFD Start relay

Grove_LCD_RGB_Backlight rgbLCD(p9, p10); //Timer display

bool cancelPending = false;
bool damperIsOpen;  
bool simulateMode = false;

I2C i2c(p9, p10);
//Serial pc(USBTX, USBRX);


//extern "C" void mbed_reset();

// Function declarations
void damperRotate(int damper);
void fullLineItoA(int m, char* str);
void countDown(int s);  //countdown in sec
void fullStop();
void debugMode();


    
int main()
{ 
    //on power up status
    cycleStartBtt.mode(PullUp);
    cycleStartBtt.set_debounce_us(1000);
    cycleStartBtt.set_samples(100);
    wait(.1);
    
    doorClosed.mode(PullUp);
    cycleStartBtt.set_debounce_us(1000);
    cycleStartBtt.set_samples(100);
    wait(.1);
    
    stopBtt.mode(PullUp);
    cycleStartBtt.set_debounce_us(1000);
    cycleStartBtt.set_samples(100);
    wait(.5);
    
    rgbLCD.clear();
    rgbLCD.setRGB(0x00, 0x66, 0xaa);                 //set the color 
    rgbLCD.locate(0,0);
    
    if(cycleStartBtt == IN_ON && stopBtt == IN_ON)
    {
        simulateMode = true; 
        rgbLCD.print("Simulating....  ");     
        wait(5);
    }
    if(stopBtt == IN_ON)
    {
       debugMode();
    }  
    
    //check for closed door before running fan
    while(doorClosed!=IN_ON)
    {
       rgbLCD.clear();
       rgbLCD.locate(0,0);
       rgbLCD.print("Close the door   ");
       wait(.25);
    }
  
    //Display startup countdown
    //Startin up and prep the tank
    rgbLCD.clear();
    rgbLCD.locate(0,0);
    rgbLCD.print("Starting up...  ");   
    
    vfd              = OUT_OFF;
    damperIsOpen = false; // Assert and run the opener once
    damperRotate(1); //vent opened;
    if(!simulateMode)
    {
        doorInterlock    = OUT_ON;
        fanRelay         = OUT_ON;
    }
    greenRelay       = OUT_OFF;
    yellowRelay      = OUT_OFF;
    redRelay         = OUT_OFF;
    
    //start timer for 90sec
    countDown(90);
    
    //Tank is ready
    rgbLCD.locate(0,0);
    rgbLCD.print("Tank is ready!  ");
    greenRelay       = OUT_ON;       //green light on
    doorInterlock    = OUT_OFF;      //interlock unlatched, ready to load material into the tank
    fanRelay         = OUT_OFF;      //fan off
    
    cancelPending = false;
   
    while(1)
    {    
        if(cancelPending)
        {
            cancelPending = false;
            
            //Display status
            greenRelay      = OUT_OFF;  //green light OFF
            yellowRelay     = OUT_OFF;  //yellow light OFF
            redRelay        = OUT_ON;   //turn red light ON
            rgbLCD.clear();
            rgbLCD.locate(0,0);
            rgbLCD.print("CYCLE STOPPED.  "); 
            vfd             = OUT_OFF;  //pump off
            if(doorClosed == IN_ON  && !simulateMode)
            {
                doorInterlock = OUT_ON; //door latched
            }
            fanRelay        = OUT_OFF;  //fan stops
            damperRotate(0);            //damper closed
            
            //wait 60sec 
            countDown(60);              
            doorInterlock=OUT_OFF;      //door unlatched
            rgbLCD.locate(0,0);
            rgbLCD.print("Tank is ready!  ");
        }
    
        if(doorClosed==IN_ON)    //door closed
        {
            /*
             *  Get set up for the wash, and wait for the start button
             */ 
            rgbLCD.locate(0,1);
            rgbLCD.print("Door closed     ");
            greenRelay      = OUT_ON;       //green light stays on
            yellowRelay     = OUT_OFF;      //yellow light off
            redRelay        = OUT_OFF;      //red light off           
            doorInterlock   = OUT_OFF;      //door unlocked
            fanRelay        = OUT_OFF;                            
            while(cycleStartBtt != IN_ON && !(doorClosed == IN_OFF || cancelPending))
            {
                if(stopBtt == IN_ON)
                {
                    cancelPending = true;
                    continue;    
                }
                wait(.25);
            }
            if(doorClosed == IN_OFF || cancelPending)
            {
                continue;
            }
            
                        
            /*
             * Cycle start button pressed - wash for 2 minutes
             */
            rgbLCD.clear();
            rgbLCD.locate(0,0);
            rgbLCD.print("CYCLE STARTED!  ");     
            greenRelay      = OUT_OFF;  //green light off
            yellowRelay     = OUT_ON;   //yellow light on                           
            damperRotate(0);            //vent close
            fanRelay        = OUT_OFF;  //fan stop
            if(!simulateMode)
            {
                doorInterlock   = OUT_ON;   //door locked
                vfd             = OUT_ON;   //vfd for pump up to speed
            }
            countDown(120);             //start timer for 2min
            if(doorClosed == IN_OFF || cancelPending)
            {
                continue;
            }
            
            
            /*
             * Finished washing - turn off the pump
             */
            vfd = OUT_OFF;              //pump off
            rgbLCD.clear();
            rgbLCD.locate(0,0);
            rgbLCD.print("CYCLE FINISHED! ");           
            wait(1);
            if(doorClosed == IN_OFF || cancelPending)
            {
                continue;
            }
            
            
            /*
             * Drip dry after washing for 5 minutes
             */
            rgbLCD.clear();
            rgbLCD.locate(0,0);
            rgbLCD.print("DRIP DRYING!    ");                
            countDown(300);
            if(doorClosed == IN_OFF || cancelPending)
            {
                continue;
            }
    
    
            /*
             * Change the air out for 2 minutes
             */
            rgbLCD.clear();
            rgbLCD.locate(0,0);
            rgbLCD.print("AIR CYCLING!    ");                                 
            damperRotate(1);            //vent opened
            if(!simulateMode)
            {
                fanRelay = OUT_ON;          //fan on
            }
            countDown(120);  
            if(doorClosed == IN_OFF || cancelPending)
            {
                continue;
            }            
            
            
            /*
             * Entire cycle is done, close up and go back to waiting
             */             
            greenRelay      = OUT_ON;       //green light on
            yellowRelay     = OUT_OFF;      //yellow light off
            doorInterlock   = OUT_OFF;      //door unlocked
            fanRelay        = OUT_OFF;      //fan off
            damperRotate(0);            //damperRelay=0;      //vent closed
            rgbLCD.clear();
            rgbLCD.locate(0,0);
            rgbLCD.print("Tank is ready!  ");
        }
        else                    //door open
        {
            rgbLCD.locate(0,1);
            rgbLCD.print("Door open       ");
            greenRelay  = OUT_OFF;  //green light OFF
            redRelay    = OUT_ON;   //turn red light ON
            yellowRelay = OUT_OFF;
            fanRelay    = OUT_OFF;  //fan stops      
            damperRotate(1);        //damperRelay=1;      //damper closed            
        }
            
        wait(1);
    }
}




void damperRotate(int damper)   //damper condition
{
    if(damper==0 && damperIsOpen == true && !simulateMode)
    {
        damperRelayAlt=OUT_ON;  //toggle damper relay
        rgbLCD.locate(0,1);
        rgbLCD.print("Closing damper  ");
        wait(5);
        damperRelayAlt=OUT_OFF; //time out to reduce wear   
        damperIsOpen = false;
    }
    else if(damper == 1 && damperIsOpen == false && !simulateMode)
    {
        damperRelay=OUT_ON;     //toggle damper relay
        rgbLCD.locate(0,1);
        rgbLCD.print("Opening damper  ");
        wait(5);
        damperRelay=OUT_OFF;    //time out to reduce wear  
        damperIsOpen = true;
    }
}

void fullLineItoA(int m, char* str)
{
    if(str == NULL)
    {
        return;
    }
    
    // Empty the array to 16 white spaces
    for(int j=0; j < 16; j++)
    {
        str[j] = ' ';   
    }
    str[16] = 0;

    // Put our counter in it, with the least-significant digit in col 4.
    int i = 3;       
    do
    {
        str[i] = m%10 + '0';
        m /= 10;
        i--;
    }while(m > 0 && i >= 0);
}

void countDown(int s)  //countdown in sec
{
    int m=0;
    char count[12];  // make a 'count' string

    for(m=s; m>0; m--)
    {
        if(stopBtt == IN_ON)
        {
           cancelPending = true;
           break;
        }

        fullLineItoA(m, count);

        rgbLCD.locate(0,1);
        rgbLCD.print(count);
        
        wait(1); 
    }
}


void debugMode()
{
    char displayLine1[17];
    char displayLine2[17];
    unsigned int count = 0;

    rgbLCD.clear();
    rgbLCD.locate(0,0);
    
    do
    {
        strcpy(displayLine1, "Door= ");
        if(doorClosed == 1)
        {
            strcat(displayLine1, "1");
        }
        else
        {
            strcat(displayLine1, "0");
        }
    
        strcat(displayLine1, ",Start=");
        if(cycleStartBtt == 1)
        {
            strcat(displayLine1, "1 ");
        }
        else
        {
            strcat(displayLine1, "0 ");
        }
    
        
        strcpy(displayLine2, "Stop= ");
        if(stopBtt == 1)
        {
            strcat(displayLine2, "1 ");
        }
        else
        {
            strcat(displayLine2, "0 ");
        }   
               
        // Every 2s, change the outputs
        if((count&7) == 0)
        {
            doorInterlock   = OUT_OFF;      //Door interlock
            fanRelay        = OUT_OFF;      //Fan control
            redRelay        = OUT_OFF;      //Red light 
            yellowRelay     = OUT_OFF;      //Yellow light 
            greenRelay      = OUT_OFF;      //Green light 
            damperRelay     = OUT_OFF;      //Damper solenoid control
            damperRelayAlt  = OUT_OFF;      //Damper solenoid toggle control                    
            vfd             = OUT_OFF;      //VFD Start relay
            
            switch((count>>3)%8)
            {
            case 0:
                doorInterlock   = OUT_ON;      //Door interlock
                strcat(displayLine2, " Lock   ");
                break;
            case 1:
                fanRelay        = OUT_ON;      //Fan control
                strcat(displayLine2, " Fan    ");
                break;
            case 2:
                redRelay        = OUT_ON;      //Red light 
                strcat(displayLine2, " Red    ");
                break;
            case 3:
                yellowRelay     = OUT_ON;      //Yellow light 
                strcat(displayLine2, " Yellow ");
                break;
            case 4:
                greenRelay      = OUT_ON;      //Green light 
                strcat(displayLine2, " Green  ");
                break;
            case 5:
                damperRelay     = OUT_ON;      //Damper solenoid control
                strcat(displayLine2, " Damper1");
                break;
            case 6:
                damperRelayAlt  = OUT_ON;      //Damper solenoid toggle control                    
                strcat(displayLine2, " Damper2");
                break;
            case 7:
//                vfd             = OUT_ON;      //VFD Start relay
                strcat(displayLine2, "        ");
                break;
            }
        }
        
        rgbLCD.locate(0,0);
        rgbLCD.print(displayLine1);
        rgbLCD.locate(0,1);
        rgbLCD.print(displayLine2);
        
        wait(.25);
        count++;
    }
    while(cycleStartBtt == IN_OFF || stopBtt == IN_OFF || doorClosed == IN_OFF);    
}