/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO Animation SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#include "MicroBit.h"
#include "Animator.h"
#include <math.h>

MicroBit uBit;

Animator my_animator(uBit.display);

void animationLoop() {
    while (my_animator.playing == 1) {
        switch (my_animator.current_animation) {
          case ANIMATION_TYPE_FLASH:
            my_animator.flash();
            break;
          case ANIMATION_TYPE_RIPPLE:
            my_animator.ripple();
            break;
          case ANIMATION_TYPE_SPIRAL:
            my_animator.spiral();
            break;
        }
        fiber_sleep(my_animator.sleep_time);    
    }    
}

void onConnected(MicroBitEvent)
{
    uBit.display.print("C");
}

void onDisconnected(MicroBitEvent)
{
    uBit.display.print("D");
    my_animator.stop();
}

void animationControl(MicroBitEvent e)
{
    if (e.value == ANIMATION_CONTROL_START) {
        my_animator.start();
        create_fiber(animationLoop);        
        return;
    }    
    if (e.value == ANIMATION_CONTROL_STOP) {
        my_animator.stop();
        return;
    }    
}

void animationType(MicroBitEvent e)
{
    my_animator.setAnimationType(e.value);
}

void onButton(MicroBitEvent e)
{
    if (e.source == MICROBIT_ID_BUTTON_B) {
          uBit.display.scroll(uBit.getName());
        return;
    }
}
    
int main()
{
    // Initialise the micro:bit runtime.
    uBit.init();

    // Insert your code here!
    uBit.display.scroll("A");

    uBit.messageBus.listen(MICROBIT_ID_BLE, MICROBIT_BLE_EVT_CONNECTED, onConnected);
    uBit.messageBus.listen(MICROBIT_ID_BLE, MICROBIT_BLE_EVT_DISCONNECTED, onDisconnected);

    uBit.messageBus.listen(ANIMATION_CONTROL_EVENT, MICROBIT_EVT_ANY, animationControl);
    uBit.messageBus.listen(ANIMATION_TYPE_EVENT,    MICROBIT_EVT_ANY, animationType);

    uBit.messageBus.listen(MICROBIT_ID_BUTTON_B, MICROBIT_BUTTON_EVT_CLICK, onButton);

/* NOTES
 * -----
 *
 * Add #include "MicroBitAnimationService.h" to MicroBitBLEManager.h
 *
 * In MicroBitConfig.h:
 *   #define MICROBIT_BLE_OPEN                       1
 *   #define MICROBIT_SD_GATT_TABLE_SIZE             0x500
 *
 * See https://developer.mbed.org/teams/Bluetooth-Low-Energy/code/BLE_GATT_Example/ for additional information on mbed support for Bluetooth low energy
 *
 */
    new MicroBitAnimationService(*uBit.ble);

    // If main exits, there may still be other fibers running or registered event handlers etc.
    // Simply release this fiber, which will mean we enter the scheduler. Worse case, we then
    // sit in the idle task forever, in a power efficient sleep.
    release_fiber();
}

