/*
The MIT License (MIT)
 
Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.
 
Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO Animation SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/
 
/**
  * Class definition for a MicroBit BLE Animation Service.
  * Provides a BLE gateway onto an Animation Model.
  */
 
#include "MicroBitConfig.h"
#include "MicroBitAnimationService.h"
#include "ble/UUID.h"
#include "Animator.h"
 
/**
  * Constructor.
  * Create a representation of the AnimationService
  * @param _ble The instance of a BLE device that we're running on.
  */
MicroBitAnimationService::MicroBitAnimationService(BLEDevice &_ble) :
        ble(_ble)
{
    // TODO: Three GattCharacteristic objects. UUID, valuePTR, length, max length, properties.
    GattCharacteristic  animationTypeCharacteristic(
        MicroBitAnimationServiceAnimationTypeCharacteristicUUID, 
        (uint8_t *)animation_type_buffer, 
        1, 
        1,
        GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE);
        
    GattCharacteristic  animationStatusCharacteristic(
        MicroBitAnimationServiceAnimationStatusCharacteristicUUID, 
        (uint8_t *)animation_status_buffer, 
        1, 
        1,
        GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);

    GattCharacteristic  animationControlCharacteristic(
        MicroBitAnimationServiceAnimationControlCharacteristicUUID, 
        (uint8_t *)animation_control_buffer, 
        1, 
        1,
        GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE);
         
    // TODO: initialise characteristic value buffers
    animation_type_buffer[0]    = 0x00;
    animation_status_buffer[0]  = 0x00;
    animation_control_buffer[0] = 0x00;
     
    // Set default security requirements - optional
    // animationTypeCharacteristic.requireSecurity(SecurityManager::MICROBIT_BLE_SECURITY_LEVEL);
    // animationStatusCharacteristic.requireSecurity(SecurityManager::MICROBIT_BLE_SECURITY_LEVEL);
    // animationControlCharacteristic.requireSecurity(SecurityManager::MICROBIT_BLE_SECURITY_LEVEL);
 
    // TODO: create GattService containing the three characteristics
    // create an array of our characteristics so we can pass them to the service when we create it
    GattCharacteristic *characteristics[] = {&animationTypeCharacteristic, &animationStatusCharacteristic, &animationControlCharacteristic};

    // create the animation service and specify its characteristics. 3rd arg is the number of characteristics.
    GattService         service(MicroBitAnimationServiceUUID, characteristics, sizeof(characteristics) / sizeof(GattCharacteristic *));
 
 
    // TODO: add the service to the Bluetooth attribute table
   ble.addService(service);
    
    // TODO: take a note of the handles of our new characteristics now that they're in the attribute table
    animationTypeCharacteristicHandle = animationTypeCharacteristic.getValueHandle();
    animationStatusCharacteristicHandle = animationStatusCharacteristic.getValueHandle();
    animationControlCharacteristicHandle = animationControlCharacteristic.getValueHandle();
    
    // TODO: register a callback function for when a characteristic is written to
    ble.onDataWritten(this, &MicroBitAnimationService::onDataWritten);
        
    // subscribe to animation status events so they can be notified to the connected client
    if (EventModel::defaultEventBus)
        EventModel::defaultEventBus->listen(ANIMATION_STATUS_EVENT, MICROBIT_EVT_ANY, this, &MicroBitAnimationService::animationStatusUpdate,  MESSAGE_BUS_LISTENER_IMMEDIATE);
}
 
 
/**
  * Callback. Invoked when any of our attributes are written to
  */
void MicroBitAnimationService::onDataWritten(const GattWriteCallbackParams *params)
{
   
    // TODO: handle write to animation type characteristic
    if (params->handle == animationTypeCharacteristicHandle && params->len == 1) {
        uint8_t *value = (uint8_t *)params->data;
        uint16_t event_value = static_cast<uint16_t>(value[0]);
        MicroBitEvent evt(ANIMATION_TYPE_EVENT, event_value);
        return;
    }

    // TODO: handle write to animation control characteristic
    if (params->handle == animationControlCharacteristicHandle && params->len == 1) {
        uint8_t *value = (uint8_t *)params->data;
        uint16_t event_value = static_cast<uint16_t>(value[0]);
        MicroBitEvent evt(ANIMATION_CONTROL_EVENT, event_value);
        return;
    }
}
 
void MicroBitAnimationService::animationStatusUpdate(MicroBitEvent e)
{
    // TODO: notify connected client of change to the animation status characteristic
    if (ble.getGapState().connected)
    {
        animation_status_buffer[0] = e.value;
        ble.gattServer().notify(animationStatusCharacteristicHandle,(uint8_t *)animation_status_buffer, 1);
    }
}
 
const uint8_t  MicroBitAnimationServiceUUID[] = {
    0xe9,0x5d,0x71,0x70,0x25,0x1d,0x47,0x0a,0xa0,0x62,0xfa,0x19,0x22,0xdf,0xa9,0xa8
};
 
const uint8_t  MicroBitAnimationServiceAnimationTypeCharacteristicUUID[] = {
    0xe9,0x5d,0xC3,0x06,0x25,0x1d,0x47,0x0a,0xa0,0x62,0xfa,0x19,0x22,0xdf,0xa9,0xa8
};
 
const uint8_t  MicroBitAnimationServiceAnimationStatusCharacteristicUUID[] = {
    0xe9,0x5d,0x45,0x92,0x25,0x1d,0x47,0x0a,0xa0,0x62,0xfa,0x19,0x22,0xdf,0xa9,0xa8
};
 
const uint8_t  MicroBitAnimationServiceAnimationControlCharacteristicUUID[] = {
    0xe9,0x5d,0xb8,0x4c,0x25,0x1d,0x47,0x0a,0xa0,0x62,0xfa,0x19,0x22,0xdf,0xa9,0xa8
};
 