/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "BLEDevice.h"
#include "ButtonService.h"

#define DEF_BUTTON_1 0x01
#define DEF_BUTTON_2 0x02
#define DEF_BUTTON_3 0x04
#define DEF_BUTTON_4 0x08

#define DEF_BUTTON_NUMBER DEF_BUTTON_1
#define DEF_BUTTON_DOWN 0x10

BLEDevice   ble;

// "BBC MICROBIT P [1-4]
const static char MSG[] = "-... -... -.-. -- .. -.-. -.- --- -... .. - .--. .---- ";
//const static char MSG[] = "-... -... -.-. -- .. -.-. -.- --- -... .. - .--. ..--- ";
//const static char MSG[] = "-... -... -.-. -- .. -.-. -.- --- -... .. - .--. ...-- ";
//const static char MSG[] = "-... -... -.-. -- .. -.-. -.- --- -... .. - .--. ....- ";

int size = sizeof(MSG)-1;
int i = 0;
int j = 0;

DigitalOut  led1(LED1);
DigitalOut  led2(LED2);

InterruptIn buttonA(BUTTON1);
InterruptIn buttonB(BUTTON2);

const static char     DEVICE_NAME[] = "BBC MicroBit vigov";
static const uint16_t uuid16_list[] = {ButtonServiceA::BUTTON_SERVICE_UUID, ButtonServiceB::BUTTON_SERVICE_UUID};

ButtonServiceA *buttonServicePtrA;
ButtonServiceB *buttonServicePtrB;

void buttonPressedCallbackA(void)
{
    buttonServicePtrA->updateButtonState(DEF_BUTTON_NUMBER | DEF_BUTTON_DOWN);
}

void buttonReleasedCallbackA(void)
{
    buttonServicePtrA->updateButtonState(DEF_BUTTON_NUMBER);
}

void buttonPressedCallbackB(void)
{
    buttonServicePtrB->updateButtonState(DEF_BUTTON_NUMBER | DEF_BUTTON_DOWN);
}

void buttonReleasedCallbackB(void)
{
    buttonServicePtrB->updateButtonState(DEF_BUTTON_NUMBER);
}

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    ble.startAdvertising();
}

void dotDash(int dash)
{
    led1 = 1;
    if(dash)
        led2 = 1;
    else
        led2 = 0;
}

void ledOff()
{
    led1 = 0;
    led2 = 0;
}

void periodicCallback(void)
{
    if(j>size)  {
        j = 0;
    }

    if (MSG[j] != ' ')  {
        if(i == 0) {
            if(MSG[j] == '-') {
                dotDash(1);
            } else {
                dotDash(0);
            }
        } else {
            ledOff();
            j++;
            i = 0;
            return;
        }
    } else {
        if(i == 4) {
            j++;
            i = 0;
            return;
        }
    }

    i++;
}

int main(void)
{
    led1 = 0;
    led2 = 0;
    Ticker ticker;
    ticker.attach(periodicCallback, 0.2);

    buttonA.fall(buttonPressedCallbackA);
    buttonA.rise(buttonReleasedCallbackA);

    buttonB.fall(buttonPressedCallbackB);
    buttonB.rise(buttonReleasedCallbackB);

    ble.init();
    ble.onDisconnection(disconnectionCallback);

    ButtonServiceA buttonServiceA(ble, DEF_BUTTON_NUMBER); /* initial value for button pressed */
    buttonServicePtrA = &buttonServiceA;

    ButtonServiceB buttonServiceB(ble, DEF_BUTTON_NUMBER); /* initial value for button pressed */
    buttonServicePtrB = &buttonServiceB;

    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));

    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);

    ble.setAdvertisingInterval(1000); /* 1000ms. */
    ble.startAdvertising();

    while (true) {
        ble.waitForEvent();
    }
}
