#include "mbed.h"

#define MAX_BITS 100
#define WIEGAND_WAIT_TIME 3000      // wait for 3000us for the next Wiegand signal
#define DATA0_PIN PTC2              // see http://developer.mbed.org/platforms/FRDM-K64F/ for pin locations
#define DATA1_PIN PTC3              // customize for board as appropriate.
#define DATA0_OUT_PIN PTC7
#define DATA1_OUT_PIN PTC5

Serial pc(USBTX, USBRX);
Timeout wiegandTimer;

InterruptIn sw(SW2);
InterruptIn data0_int(DATA0_PIN);
InterruptIn data1_int(DATA1_PIN);
DigitalOut data0_out(DATA0_OUT_PIN);
DigitalOut data1_out(DATA1_OUT_PIN);

volatile unsigned char dataBits[MAX_BITS];  // stores all of the data bits
volatile unsigned char bitCount = 0;        // number of bits captured
volatile bool flagDataIn = false;           // false when data is currently being captured
volatile bool readComplete = false;         // true when a card read has been completed
volatile bool sendData = false;             

void txWiegand(uint8_t *data, uint8_t size) {
    // sends wiegand data for testing purposes. 
    for (uint8_t i=0; i<size; i++) {
        // use a ternary operator to toggle the correct pin
        data[i] ? data1_out=0 : data0_out=0; 
        wait_us(40);
        data[i] ? data1_out=1 : data0_out=1;
        wait_us(2249);
    }
} 

void parseWiegand (void) {
    // tell the main while loop the signal to process data
    readComplete = true;
}

void DATA0_ISR() {
    bitCount++;
    flagDataIn = true;
}

void DATA1_ISR() {
    dataBits[bitCount] = 1;
    bitCount++;
    flagDataIn = true;
}

void sw_isr() {
    sendData = true;        
}

int main()
{
    pc.baud (115200);
    
    data0_int.fall(&DATA0_ISR); // assign ISRs 
    data1_int.fall(&DATA1_ISR);
    sw.fall(&sw_isr);
       
    data0_out = 1;
    data1_out = 1;
    
    while (true) {
        if ((bitCount > 0) && flagDataIn) {
            // we got more valid data so reset the timer
            wiegandTimer.attach_us(&parseWiegand, WIEGAND_WAIT_TIME);
            flagDataIn = false;
        }
        if (readComplete) {
            // After the Wiegand data has been read we need to parse it
            pc.printf("\nRead %d bits: ", bitCount);
            // Print the card value in binary
            for (uint8_t i=0; i<bitCount; i++) {
                pc.printf("%d", dataBits[i]);
            }
            pc.printf("\n");
            
            /* in the future parse card code and facility code here.
             */
            
            // reset variables for next read
            readComplete = false;
            bitCount = 0;
            for (uint8_t i=0; i<MAX_BITS; i++) {
                dataBits[i] = 0;
            }
        }
        if (sendData) {
            // do things
            uint8_t testData[26] = { 0,0,1,1,0,0,1,0,0,0,0,1,0,0,1,1,0,0,0,1,0,1,0,1,0,0 };
            txWiegand(testData, 26); 
            sendData = false;
        }  
    }
}