
//
// nlg_mini.cpp
//

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "nlg_mini.h"

/* 単位を扱いやすいように */
typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned long dword;

// 変数読み出し(WORD)
word ReadWORD(byte *p)
{
    return
        ((word)p[0]) |
        ((word)p[1])<<8;
}

// 変数読み出し(DWORD)
dword ReadDWORD(byte *p)
{
    return
        ((dword)p[0]) |
        ((dword)p[1])<<8 |
        ((dword)p[2])<<16 |
        ((dword)p[3])<<24;
}

// NLGファイルを開く
int OpenNLG(NLG_CTX *np, const char *filename)
{
    byte hdr[0x60];

    np->fp = fopen(filename, "rb");

    // ファイルが開けない
    if (!np->fp)
        return NLG_FILEERR;

    // ヘッダ部分の読み込み
    fread(hdr, 0x60, 1, np->fp);

    // IDの確認
    if (memcmp(hdr, "NLG1", 4) != 0)
    {
        CloseNLG(np);
        return NLG_UNK_FORMAT;
    }

    // バージョン
    np->version = ReadWORD(hdr + 4);

    // クロック
    np->baseclk = ReadDWORD(hdr + 72);

    // ティック
    np->tick = ReadDWORD(hdr + 76);
    np->tick_us = np->tick / 4;

    // 長さ
    np->length = ReadDWORD(hdr + 88);

    // 位置
    fseek(np->fp, 0x60, SEEK_SET);

    // CTC初期化
    np->ctc0 = np->ctc3 = 0;

    return NLG_OK;
}

// ファイルを閉じる
void CloseNLG(NLG_CTX *np)
{
    if (!np->fp)
        return;

    fclose(np->fp);

#if defined(__MICROLIB) && defined(__ARMCC_VERSION) // with microlib and ARM compiler
    free(np->fp);
#endif

    np->fp = NULL;
}

// データの読み出し
int ReadNLG(NLG_CTX *np)
{
    return fgetc(np->fp);
}

// ファイルポインタの位置を取得
long TellNLG(NLG_CTX *np)
{
    return ftell(np->fp);
}

// ファイルポインタの位置を設定
void SeekNLG(NLG_CTX *np, long pos)
{
    fseek(np->fp, pos, SEEK_SET);
}

// ティック(us)の取得
int GetTickUsNLG(NLG_CTX *np)
{
    return np->tick_us;
}

// Set Tick
// (CTC0 * 64us) * CTC3
static inline void SetTickNLG(NLG_CTX *np)
{
    np->tick = ((np->ctc0 * 256) * np->ctc3);
    np->tick_us = np->tick / 4;
}

// CTC0値の設定
void SetCTC0_NLG(NLG_CTX *np, int value)
{
    np->ctc0 = value;
    SetTickNLG(np);
}

// CTC3値の設定
void SetCTC3_NLG(NLG_CTX *np, int value)
{
    np->ctc3 = value;
    SetTickNLG(np);
}

// 曲の長さを得る
int GetLengthNLG(NLG_CTX *np)
{
    return np->length;
}

// ベースクロックを得る
int GetBaseClkNLG(NLG_CTX *np)
{
    return np->baseclk;
}
