#include "Lora.h"
#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>

Lora::Lora()
{
    // get a mDot handle
    dot = mDot::getInstance();
    logInfo("version: %s", dot->getId().c_str());
    //joinNetwork();
}



bool Lora::setting(string config_network_name, string config_network_pass,uint8_t config_frequency_sub_band,uint8_t config_ACK)
{
    int32_t ret;
    bool state = true;
    //*******************************************
    // configuration
    //*******************************************
    // reset to default config so we know what state we're in
    dot->resetConfig();
    
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    // set up the mDot with our network information: frequency sub band, network name, and network password
    // these can all be saved in NVM so they don't need to be set every time - see mDot::saveConfig()
    
    // frequency sub band is only applicable in the 915 (US) frequency band
    // if using a MultiTech Conduit gateway, use the same sub band as your Conduit (1-8) - the mDot will use the 8 channels in that sub band
    // if using a gateway that supports all 64 channels, use sub band 0 - the mDot will use all 64 channels
   
    logInfo("setting frequency sub band");
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) 
    {
        logError("failed to set frequency sub band %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        state = false;
    }
    
    logInfo("setting network name");
    if ((ret = dot->setNetworkName(config_network_name)) != mDot::MDOT_OK) 
    {
        logError("failed to set network name %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        state = false;
    }
    
    logInfo("setting network password");
    if ((ret = dot->setNetworkPassphrase(config_network_pass)) != mDot::MDOT_OK) 
    {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        state = false;
    }
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    logInfo("setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::SF_10)) != mDot::MDOT_OK) 
    {
        logError("failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        state = false;
    }
    
    // request receive confirmation of packets from the gateway
    logInfo("enabling ACKs");
    if ((ret = dot->setAck(config_ACK)) != mDot::MDOT_OK) 
    {
        logError("failed to enable ACKs %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        state = false;
    }
    
     // set join mode to AUTO_OTA so the mDot doesn't have to rejoin after sleeping   
    logInfo("setting join mode to AUTO_OTA");
    if ((ret = dot->setJoinMode(mDot::AUTO_OTA)) != mDot::MDOT_OK) {
        logError("failed to set join mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("saving config");
    if (! dot->saveConfig()) 
    {
        logError("failed to save configuration");
        state = false;
    }
    //*******************************************
    // end of configuration
    //*******************************************
    return state;
}

bool Lora::joinNetwork()
{
     int32_t ret;
     // attempt to join the network
    if (!dot->getNetworkJoinStatus()) 
    {    
        logInfo("joining network");
        while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) 
        {
            logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
            // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
            osDelay((uint32_t)dot->getNextTxMs());
        } 
    }
    
    return true;   
}

bool Lora::sendData_vector(const std::vector<uint8_t>& data)
{
    int32_t ret;
    if ((ret = dot->send(data)) != mDot::MDOT_OK) 
    {
        logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        return false;
    } 
    else 
    {
        logInfo("successfully sent data to gateway");
        return true;
    }    
}

bool Lora::sendData_string(std::string& input_data)
{
     int32_t ret;
    vector<uint8_t> sent_data;
    for (std::string::iterator it = input_data.begin(); it != input_data.end(); it++)
    {
        sent_data.push_back((uint8_t) *it);   
    }
    
    if ((ret = dot->send(sent_data)) != mDot::MDOT_OK) 
    {
            logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
    } 
    else 
    {
            logInfo("successfully sent data to gateway");
    }
    return true;
    
}

 bool Lora::receiveData_string(std::string& output_data)
{
    int32_t ret;
    vector<uint8_t> receive_data;
     if ((ret = dot->recv(receive_data)) != mDot::MDOT_OK) 
    {
            logError("failed to recv: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
            return false;
    } 
    else 
    {
            string str(receive_data.begin(),receive_data.end());
            output_data=str;
    }
    return true;
           
}