#ifndef __included_fast_analog_in_h
#define __included_fast_analog_in_h

#include "mbed.h"

namespace NK
{

using namespace mbed;
extern "C" void ADC0_IRQHandler(void);

class FastAnalogIn: public mbed::AnalogIn
{
public:
    void start_read() {
        ADC0->SC1[0] = ADC_SC1_ADCH(_adc.adc)
                       | ADC_SC1_AIEN_MASK;     // enable interrupt
    }

    void abort_read() {
        ADC0->SC1[0] = ADC_SC1_ADCH(0x1F);
    }

    uint16_t read_u16() {
        // Wait Conversion Complete
        while ((ADC0->SC1[0] & ADC_SC1_COCO_MASK) != ADC_SC1_COCO_MASK);

        // Return value
        return (uint16_t)ADC0->R[0];
    }

    uint16_t read_u16_nowait() {
        return (uint16_t)ADC0->R[0];
    }

    // 0x0000 => 0x80 (-128)
    // 0x7fff => 0xFF (-1)
    // 0x8000 => 0x00 (0)
    // 0xffff => 0x7f (127)
    int8_t read_s8_nowait() {
        int32_t val = read_u16_nowait();
        return (val + 0x80 - 0x8000) >> 8;
    }

    FastAnalogIn(PinName pin)
        : mbed::AnalogIn(pin) {

        NVIC_SetVector(ADC0_IRQn, (uint32_t)&ADC0_IRQHandler);
        NVIC_DisableIRQ(ADC0_IRQn);

        ADC0->CFG1 = ADC_CFG1_ADLPC_MASK    // Low-Power Configuration
                     | ADC_CFG1_ADIV(4)       // Clock Divide Select: (Input Clock)/4
                     | ADC_CFG1_ADLSMP_MASK   // Long Sample Time
                     | ADC_CFG1_MODE(3)       // (16)bits Resolution
                     | ADC_CFG1_ADICLK(1);    // Input Clock: (Bus Clock)/2

        ADC0->CFG2 = 0   // ADxxA channels are selected
                     | ADC_CFG2_ADACKEN_MASK    // Asynchronous Clock Output Enable
                     | ADC_CFG2_ADHSC_MASK      // High-Speed Configuration
                     | ADC_CFG2_ADLSTS(0);      // Long Sample Time Select

        ADC0->SC2 = ADC_SC2_REFSEL(0);      // Default Voltage Reference

        ADC0->SC3 = ADC_SC3_AVGE_MASK       // Hardware Average Enable
                    | ADC_SC3_AVGS(3);        // 16 Samples Averaged

        ADC0->SC1[0] = ADC_SC1_ADCH(_adc.adc)
                       | ADC_SC1_AIEN_MASK;     // enable interrupt
    }

    void enable_interrupt() {
        NVIC_EnableIRQ(ADC0_IRQn);
    }
    void disable_interrupt() {
        NVIC_DisableIRQ(ADC0_IRQn);
    }

};

}

#endif