#ifndef __included_audio_analyzer_h
#define __included_audio_analyzer_h

#include <math.h>

namespace NK
{

class AudioAnalyzer
{
protected:
    int8_t const *samples;
    uint16_t nsamples;
    uint16_t zeroCrossings;
    uint32_t power;
    float logPower;
    float powerRef;
    int8_t minValue;
    int8_t maxValue;
    bool analyzed;

    void analyze();

public:
    AudioAnalyzer(int8_t const *_samples, uint16_t _nsamples)
        : samples(_samples), nsamples(_nsamples), zeroCrossings(0), power(0), logPower(0.0), powerRef(0.0), analyzed(false) {
    }

    uint16_t getZeroCrossings() {
        if (!analyzed) analyze();
        return zeroCrossings;
    }

    float getZeroCrossingRatioPercent() {
        return getZeroCrossings() * 100.0 / nsamples;
    }

    uint32_t getPower() {
        if (!analyzed) analyze();
        return power;
    }

    float getLogPower() {
        if (!analyzed) analyze();
        logPower = ::log((double)power) - powerRef;
        return logPower;
    }

    void getMinMaxValues(int8_t *min, int8_t *max) {
        if (!analyzed) analyze();
        *min = minValue;
        *max = maxValue;
    }

    bool isVoiced() {
        return !(isnan(getLogPower()) || logPower < PowerThreshold);
    }

    void setPowerRef(float _powerRef) {
        powerRef = _powerRef;
    }

    // anything with logPower above PowerThreshold
    // and below the line
    // zeroCrossingRatioPercent = VowelSlope * logPower + VowelIntercept
    bool isVowel() {
        getLogPower();
        if (logPower < PowerThreshold)
            return false;
        return (getZeroCrossingRatioPercent() < VowelSlope * (logPower - VowelXIntercept));
    }

    static const float PowerThreshold = -4.0;
    // anything below the line
    // zeroCrossingRatioPercent = VowelSlope * logPower + VowelIntercept
    // and above PowerThreshold
    // is considered a vowel.
    static const float VowelSlope = 14.7;
    static const float VowelXIntercept = -0.7;
};

} // namespace NK
#endif