// Mbed library to control LPD8806-based RGB LED Strips
// (c) 2011 Jelmer Tiete
// This library is ported from the Arduino implementation of Adafruit Industries
// found at: http://github.com/adafruit/LPD8806
// and their strips: http://www.adafruit.com/products/306
// Released under the MIT License: http://mbed.org/license/mit
//
// Parameterized and modified to use soft SPI.
// Jas Strong <jasmine@electronpusher.org>
/*****************************************************************************/

#include "LPD8806.h"

LPD8806::LPD8806(PinName dataPin, PinName clockPin, int n) :
    dat(dataPin),
    clk(clockPin) {
    // Allocate 3 bytes per pixel:
    numLEDs = n;
    if (NULL != (pixels = (uint8_t *)malloc(numLEDs * 3))) {
        memset(pixels, 0x80, numLEDs * 3); // Init to RGB 'off' state
    }
}

/*
 *  Soft SPI clock-out implementation (CPOL = 0, CPHA = 0).
 *  Certainly not the fastest in the world but it'll do.
 *  Gets about 3.6 MHz;  could get several times as much
 *  using the bitbands directly  - jas.
 */
 
void LPD8806::write(uint8_t byte) {
    for (int i=0; i<8; i++) {
        clk = 0;
        dat = (byte & 0x80);
        clk = 1;
        byte <<= 1;
    }
    clk = 0;
}

void LPD8806::begin(void) {

    // Issue initial latch to 'wake up' strip (latch length varies w/numLEDs)
    writezeros(3 * ((numLEDs + 63) / 64));
}

uint16_t LPD8806::numPixels(void) {
    return numLEDs;
}

void LPD8806::writezeros(uint16_t n) {
    while (n--) write(0x00);
}

void LPD8806::blank(void) {
    memset(pixels, 0x80, numLEDs * 3);
}

// This is how data is pushed to the strip.  Unfortunately, the company
// that makes the chip didnt release the  protocol document or you need
// to sign an NDA or something stupid like that, but we reverse engineered
// this from a strip controller and it seems to work very nicely!
void LPD8806::show(void) {
    uint16_t i, nl3 = numLEDs * 3; // 3 bytes per LED

    for (i=0; i<nl3; i++ ) {
        write(pixels[i]);
    }

    // Write latch at end of data; latch length varies with number of LEDs
    writezeros(3 * ((numLEDs + 63) / 64));
}

// Convert R,G,B to combined 32-bit color
uint32_t LPD8806::Color(uint8_t r, uint8_t g, uint8_t b) {
    // Take the lowest 7 bits of each value and append them end to end
    // We have the top bit set high (its a 'parity-like' bit in the protocol
    // and must be set!)
    return 0x808080 | ((uint32_t)g << 16) | ((uint32_t)r << 8) | (uint32_t)b;
}

// store the rgb component in our array
void LPD8806::setPixelColor(uint16_t n, uint8_t r, uint8_t g, uint8_t b) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3  ] = g | 0x80;
    pixels[n*3+1] = r | 0x80;
    pixels[n*3+2] = b | 0x80;
}

void LPD8806::setPixelGB(uint16_t n, uint8_t g, uint8_t b) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3  ] = g | 0x80;
    pixels[n*3+2] = b | 0x80;
}

void LPD8806::setPixelR(uint16_t n, uint8_t r) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3+1] = r | 0x80;
}

void LPD8806::setPixelG(uint16_t n, uint8_t g) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3] = g | 0x80;
}

void LPD8806::setPixelB(uint16_t n, uint8_t b) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3+2] = b | 0x80;
}

void LPD8806::setPixelColor(uint16_t n, uint32_t c) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3  ] = (c >> 16) | 0x80;
    pixels[n*3+1] = (c >>  8) | 0x80;
    pixels[n*3+2] =  c        | 0x80;
}
