// 800 KHz WS2811 driver, kinda.
//
// Parameterized and modified to use soft SPI.
// Jas Strong <jasmine@electronpusher.org>
/*****************************************************************************/

#include "LedStrip.h"
#include "WS2811.h"
#include "bitband.h"

extern int onewire_speed_multiplier;   // speed setting for ws2811 et al.


WS2811::WS2811(PinName dataPin, PinName clockPin, int n) :
    dat(dataPin),
    clk(clockPin) {
    // Allocate 3 bytes per pixel:
    numLEDs = n;
    pixels = (uint8_t *)malloc(numLEDs * 3);
    if (pixels) {
        memset(pixels, 0x00, numLEDs * 3); // Init to RGB 'off' state
    }
    // calibrate delay loops for NRZ 
    int i;
    guardtime.start();
    for (i=0; i<1000; i++)
        /* do nothing */;
    i=guardtime.read_us();
    printf("ws2811:  1000 iters took %d usec.\n", i);
    bogocal = (1000 / (onewire_speed_multiplier * i * 4.2)); // iterations per bitcell (417 nsec)
    printf("ws2811:  calibrating to %d bogojiffies.\n", bogocal);
    
    data_mask = dat.get_mask();
    clock_mask = clk.get_mask();
    data_set = dat.get_set();
    data_clr = dat.get_clr();
    clock_set = clk.get_set();
    clock_clr = clk.get_clr();
    
    printf("ws2811:          data mask 0x%x, data set reg 0x%x, data clear reg 0x%x\r\n", data_mask, data_set, data_clr);
    printf("ws2811: (unused) clock mask 0x%x, clock set reg 0x%x, clock clear reg 0x%x\r\n", clock_mask, clock_set, clock_clr);
    
}

/*
 *  These chips use a one-wire protocol based on a sort of NRZ signalling- jas.
 */
 
void WS2811::write(uint8_t byte) {
    
    for (int i=0; i<8; i++) {
        if (byte & 0x80)
            writebit(1);
        else
            writebit(0);
        byte <<= 1;
    }
}

inline void WS2811::celldelay(void) {
    for (volatile int i = 0; i<bogocal; i++)
        /* do nothing */ ;
}

inline void WS2811::writebit(bool bit) {
    // first cell is always 1
    (*data_set) = data_mask;
    celldelay();
    if (bit) {
        (*clock_set) = data_mask; // dummy, we don't care but must take constant time
        celldelay();
    } else {
        (*data_clr) = data_mask;
        celldelay();
    }
    // last cell is always 0
    (*data_clr) = data_mask;
    celldelay();
}

void WS2811::begin(void) {
    blank();
    show();
}

uint16_t WS2811::numPixels(void) {
    return numLEDs;
}

void WS2811::blank(void) {
    memset(pixels, 0x00, numLEDs * 3);
}

void WS2811::show(void) {
    uint16_t i, nl3 = numLEDs * 3; // 3 bytes per LED
    while (guardtime.read_us() < 50)
        /* spin */;
    __disable_irq();
    for (i=0; i<nl3; i++ ) {
        write(pixels[i]);
    }
    __enable_irq();
    guardtime.reset();
}


uint32_t WS2811::total_luminance(void) {
    uint32_t running_total;
    running_total = 0;
    for (int i=0; i<numLEDs*3; i++)
        running_total += pixels[i];
    return running_total;
}

// Convert R,G,B to combined 32-bit color
uint32_t WS2811::Color(uint8_t r, uint8_t g, uint8_t b) {
    // Take the lowest 7 bits of each value and append them end to end
    // We have the top bit set high (its a 'parity-like' bit in the protocol
    // and must be set!)
    return ((uint32_t)g << 16) | ((uint32_t)r << 8) | (uint32_t)b;
}

// store the rgb component in our array
void WS2811::setPixelColor(uint16_t n, uint8_t r, uint8_t g, uint8_t b) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3  ] = g;
    pixels[n*3+1] = r;
    pixels[n*3+2] = b;
}

void WS2811::setPixelR(uint16_t n, uint8_t r) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3+1] = r;
}

void WS2811::setPixelG(uint16_t n, uint8_t g) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3] = g;
}

void WS2811::setPixelB(uint16_t n, uint8_t b) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3+2] = b;
}

void WS2811::setPackedPixels(uint8_t * buffer, uint32_t n) {
    if (n >= numLEDs) return;
    memcpy(pixels, buffer, (size_t) (n*3));
}

void WS2811::setPixelColor(uint16_t n, uint32_t c) {
    if (n >= numLEDs) return; // '>=' because arrays are 0-indexed

    pixels[n*3  ] = (c >> 16);
    pixels[n*3+1] = (c >>  8);
    pixels[n*3+2] =  c;
}
