#pragma once

// define constants
//------------------------------------------------------------------------------------------------------------------
#define AFRL_Offset 0x20
#define AFRH_Offset 0x24
#define MODER_Offset 0x00
#define PUPDR_Offset 0x0C
#define OSPEEDR_Offset 0x08 
#define OTYPER_Offset 0x04
#define IDR_Offset 0x10
#define ODR_Offset 0x14
#define gpioa 0x48000000
#define gpiob 0x48000400
#define gpiof 0x48001400

// include files
//------------------------------------------------------------------------------------------------------------------
#include "mbed.h"

// macros
//------------------------------------------------------------------------------------------------------------------
#define name(var)  #var 


// assembly functions
//------------------------------------------------------------------------------------------------------------------
extern "C" int check_1_bit(uint32_t gpiox, uint32_t offset, uint32_t pin);
extern "C" int check_2_bit(uint32_t gpiox, uint32_t offset, uint32_t pin);
extern "C" int check_alternative_mode(uint32_t gpiox, uint32_t offset, uint32_t pin);

// structs
//------------------------------------------------------------------------------------------------------------------
typedef struct Pin {
    char port;
    int number;
} pin_t;


/** Debug_complete class.
 *  Class for stepping program, printing actual position and complete configuration of certain peripherals.
 *
 *
 * Example program:
 * @code
 * // ----------------------------------------------------------------------------
 * // Author: Lukas Bielesch 
 * // Department of Measurement, Czech technical university in Prague, Czech Republic 
 * // Date of publication: 15. Apr 2019
 * // ----------------------------------------------------------------------------
 * #include "Debug.h"
 * AnalogIn analog(PA_5);
 * PwmOut pwm(PA_6);
 * DigitalOut out(PA_4);
 * Debug_complete pc(PA_2, PA_3, 115200);
 *  
 * int main(){
 *     out = 1;
 *     pc.breakpoint(__LINE__); 
 *     pwm = 0.5;
 *     pwm.period(1);
 *     while(1){
 *         pwm = analog;
 *         pc.breakpoint(__LINE__);
 *         wait(0.5);
 *         pwm.period_ms(analog.read_u16()%2000);
 *         pc.breakpoint();        
 *     }
 * }
 * @endcode
 */
class Debug_complete {
public:

    /** Create object of class Debug_complete
     * @param tx_pin TX pin of debug serial port of the board
     * @param rx_pin RX pin of debug serial port of the board
     * @param baudrate desired baudrate value of serial port, default value is 115200 Bd/s
     */
    Debug_complete(PinName tx_pin, PinName rx_pin, int baudrate = 115200);
     
    /** Perform one breakpoint. Function stops the program and shows configuration of certain peripherals,  
     * then it waits until any character is received from PC and program continues
     * @param line_number line number of the breakpoint, __LINE__ is recommended to use
     */ 
    void breakpoint(int line_number = -1);
    

  
private:  
// objects
    Serial pc; //debug serial device
// variables
    int breakpoint_count; //stores number of the current breakpoint 
//functions
    // print alternate function of pin    
    void print_af_mode( char portx, int pin_number, int af_mode);
    // show configuration of all pins
    void show_all_pins_config();
    // show configuration of one pin
    void show_pin_config(pin_t pin);
    // print config of all timers
    void show_all_timers_config();
    // print configuration of timer
    void show_tim_config(int timer);    
    // print configuration of pin in pwm output mode
    void show_pwm_config(int timer, int channel);
    // print configuration of pin in analog input mode    
    void show_analog_config(int channel);
    // print configuration of adc1
    void show_adc1_config();
    //print configuration of system clock
    void show_clk_config();
    // initialization function 
    void init();
    // prit line to serial port
    void line();
};

