/***********************************************************************
This is the Quectel L80 GPS library.
It is based on the Adafruit GPS library and has been adapted in order to
work with Mbed OS.

Initially written by Limor Fried/Ladyada for Adafruit Industries.
Modified by Biagio Montaruli <biagio.hkr@gmail.com>
BSD license, check license.txt for more information
All text above must be included in any redistribution
************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "mbed.h"
#include <ctype.h>
#include "Quectel_L80_GPS.h"

/* Constructor */
Quectel_L80_GPS::Quectel_L80_GPS(Serial *ser)
{
    gpsHwSerial = ser;

    recvdflag   = false;
    paused      = false;
    lineidx     = 0;
    currentline = line1;
    lastline    = line2;
    inFullOnMode = true;
    inStandbyMode = false;
    inAlwaysLocateMode = false;

    hours = minutes = seconds = milliseconds = 0;
    year = month = day = 0;
    fixQuality = satellites = 0;
    lat = lon = mag = 0;
    fix = false;
    latitude = longitude = altitude = 0.0;
    geoidheight = speed = angle = magvariation = HDOP = 0.0;

    LOCUS_serial = LOCUS_records = 0;
    LOCUS_type = LOCUS_mode = LOCUS_config = LOCUS_interval = 0;
    LOCUS_distance = LOCUS_speed = LOCUS_status = LOCUS_percent = 0;
}

bool Quectel_L80_GPS::parse(char *nmea)
{
    if (nmea[strlen(nmea) - 4] == '*') {
        uint16_t sum = parseHex(nmea[strlen(nmea) - 3]) * 16;
        sum += parseHex(nmea[strlen(nmea) - 2]);

        /* compute checksum */
        for (uint8_t i = 2; i < (strlen(nmea) - 4); i++) {
            sum ^= nmea[i];
        }
        if (sum != 0) {
            /* bad checksum */
            return false;
        }
    }
    int32_t degree;
    long min;
    char degreebuff[10];
    /* look for a few common sentences */
    if (strstr(nmea, "$GPGGA")) {
        /* found GGA */
        char *p = nmea;
        /* get time */
        p = strchr(p, ',') + 1;
        double timef = atof(p);
        uint32_t time = timef;
        hours = time / 10000;
        minutes = (time % 10000) / 100;
        seconds = (time % 100);

        milliseconds = fmod(timef, 1.0) * 1000;

        /* parse out latitude */
        p = strchr(p, ',') + 1;
        if (',' != *p) {
            strncpy(degreebuff, p, 2);
            p += 2;
            degreebuff[2] = '\0';
            degree = atol(degreebuff) * 10000000;
            strncpy(degreebuff, p, 2);
            p += 3; /* skip decimal point */
            strncpy(degreebuff + 2, p, 4);
            degreebuff[6] = '\0';
            min = 50 * atol(degreebuff) / 3;
            latitude_fixed = degree + min;
            latitude = degree / 100000 + min * 0.000006F;
            latitudeDegrees = (latitude - 100 * int(latitude / 100)) / 60.0;
            latitudeDegrees += int(latitude / 100);
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            if (p[0] == 'S') latitudeDegrees *= -1.0;
            if (p[0] == 'N') lat = 'N';
            else if (p[0] == 'S') lat = 'S';
            else if (p[0] == ',') lat = 0;
            else return false;
        }

        /* parse out longitude */
        p = strchr(p, ',') + 1;
        if (',' != *p) {
            strncpy(degreebuff, p, 3);
            p += 3;
            degreebuff[3] = '\0';
            degree = atol(degreebuff) * 10000000;
            strncpy(degreebuff, p, 2);
            p += 3; /* skip decimal point */
            strncpy(degreebuff + 2, p, 4);
            degreebuff[6] = '\0';
            min = 50 * atol(degreebuff) / 3;
            longitude_fixed = degree + min;
            longitude = degree / 100000 + min * 0.000006F;
            longitudeDegrees = (longitude - 100 * int(longitude / 100)) / 60.0;
            longitudeDegrees += int(longitude / 100);
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            if (p[0] == 'W') longitudeDegrees *= -1.0;
            if (p[0] == 'W') lon = 'W';
            else if (p[0] == 'E') lon = 'E';
            else if (p[0] == ',') lon = 0;
            else return false;
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            fixQuality = atoi(p);
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            satellites = atoi(p);
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            HDOP = atof(p);
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            altitude = atof(p);
        }

        p = strchr(p, ',') + 1;
        p = strchr(p, ',') + 1;
        if (',' != *p) {
            geoidheight = atof(p);
        }
        return true;
    }
    if (strstr(nmea, "$GPRMC")) {
        /* found RMC */
        char *p = nmea;

        /* get time */
        p = strchr(p, ',') + 1;
        double timef = atof(p);
        uint32_t time = timef;
        hours = time / 10000;
        minutes = (time % 10000) / 100;
        seconds = (time % 100);

        milliseconds = fmod(timef, 1.0) * 1000;

        p = strchr(p, ',') + 1;
        if (p[0] == 'A')
            fix = true;
        else if (p[0] == 'V')
            fix = false;
        else
            return false;

        /* parse out latitude */
        p = strchr(p, ',') + 1;
        if (',' != *p) {
            strncpy(degreebuff, p, 2);
            p += 2;
            degreebuff[2] = '\0';
            long degree = atol(degreebuff) * 10000000;
            strncpy(degreebuff, p, 2);
            p += 3; /* skip decimal point */
            strncpy(degreebuff + 2, p, 4);
            degreebuff[6] = '\0';
            min = 50 * atol(degreebuff) / 3;
            latitude_fixed = degree + min;
            latitude = degree / 100000 + min * 0.000006F;
            latitudeDegrees = (latitude - 100 * int(latitude / 100)) / 60.0;
            latitudeDegrees += int(latitude / 100);
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            if (p[0] == 'S') latitudeDegrees *= -1.0;
            if (p[0] == 'N') lat = 'N';
            else if (p[0] == 'S') lat = 'S';
            else if (p[0] == ',') lat = 0;
            else return false;
        }

        /* parse out longitude */
        p = strchr(p, ',') + 1;
        if (',' != *p) {
            strncpy(degreebuff, p, 3);
            p += 3;
            degreebuff[3] = '\0';
            degree = atol(degreebuff) * 10000000;
            strncpy(degreebuff, p, 2);
            p += 3; /* skip decimal point */
            strncpy(degreebuff + 2, p, 4);
            degreebuff[6] = '\0';
            min = 50 * atol(degreebuff) / 3;
            longitude_fixed = degree + min;
            longitude = degree / 100000 + minutes * 0.000006F;
            longitudeDegrees = (longitude - 100 * int(longitude / 100)) / 60.0;
            longitudeDegrees += int(longitude / 100);
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            if (p[0] == 'W') longitudeDegrees *= -1.0;
            if (p[0] == 'W') lon = 'W';
            else if (p[0] == 'E') lon = 'E';
            else if (p[0] == ',') lon = 0;
            else return false;
        }
        /* speed */
        p = strchr(p, ',')+1;
        if (',' != *p) {
            speed = atof(p);
        }

        /* angle */
        p = strchr(p, ',') + 1;
        if (',' != *p) {
            angle = atof(p);
        }

        p = strchr(p, ',') + 1;
        if (',' != *p) {
            uint32_t fulldate = atof(p);
            day = fulldate / 10000;
            month = (fulldate % 10000) / 100;
            year = (fulldate % 100);
        }

        return true;
    }

    return false;
}

char Quectel_L80_GPS::read(void)
{
    char c = 0;

    if (paused) return c;

    if(!gpsHwSerial->readable()) return c;

    c = gpsHwSerial->getc();
#ifdef SERIAL_DEBUG
    pcSerial.print(c);
#endif

    if (c == '\n') {
        currentline[lineidx] = 0;

        if (currentline == line1) {
            currentline = line2;
            lastline = line1;
        } else {
            currentline = line1;
            lastline = line2;
        }

#ifdef SERIAL_DEBUG
        pcSerial.println("----");
        pcSerial.println((char *)lastline);
        pcSerial.println("----");
#endif
        lineidx = 0;
        recvdflag = true;
    }

    currentline[lineidx++] = c;
    if (lineidx >= MAXLINELENGTH)
        lineidx = MAXLINELENGTH-1;

    return c;
}

void Quectel_L80_GPS::sendCommand(const char *str)
{
    gpsHwSerial->printf("%s\n", str);
}

bool Quectel_L80_GPS::newNMEAreceived(void)
{
    return recvdflag;
}

void Quectel_L80_GPS::pause(bool p)
{
    paused = p;
}

char *Quectel_L80_GPS::lastNMEA(void)
{
    recvdflag = false;
    return (char *)lastline;
}

/* read a Hex value and return the decimal equivalent */
uint8_t Quectel_L80_GPS::parseHex(char c)
{
    if (c < '0')
        return 0;
    if (c <= '9')
        return c - '0';
    if (c < 'A')
        return 0;
    if (c <= 'F')
        return (c - 'A') + 10;
    // if (c > 'F')
    return 0;
}

bool Quectel_L80_GPS::waitForSentence(const char *waitstr, uint8_t max)
{
    char str[20];

    uint8_t i = 0;
    while (i < max) {
        read();

        if (newNMEAreceived()) {
            char *nmea = lastNMEA();
            strncpy(str, nmea, 20);
            str[19] = 0;
            i++;

            if (strstr(str, waitstr))
                return true;
        }
    }

    return false;
}

uint8_t Quectel_L80_GPS::getHour(void)
{
    return hours;
}

uint8_t Quectel_L80_GPS::getMinutes(void)
{
    return minutes;
}

uint8_t Quectel_L80_GPS::getSeconds(void)
{
    return seconds;
}

uint8_t Quectel_L80_GPS::getMilliseconds(void)
{
    return milliseconds;
}

uint8_t Quectel_L80_GPS::getDay(void)
{
    return day;
}

uint8_t Quectel_L80_GPS::getMonth(void)
{
    return month;
}

uint8_t Quectel_L80_GPS::getYear(void)
{
    return year;
}

float Quectel_L80_GPS::getLatitude(void)
{
    return latitude;
}

float Quectel_L80_GPS::getLongitude(void)
{
    return longitude;
}

int32_t Quectel_L80_GPS::getLatitudeFixed(void)
{
    return latitude_fixed;
}

int32_t Quectel_L80_GPS::getLongitudeFixed(void)
{
    return longitude_fixed;
}

float Quectel_L80_GPS::getLatitudeDegrees(void)
{
    return latitudeDegrees;
}

float Quectel_L80_GPS::getLongitudeDegrees(void)
{
    return longitudeDegrees;
}

float Quectel_L80_GPS::getAltitude(void)
{
    return altitude;
}

float Quectel_L80_GPS::getGeoidheight(void)
{
    return geoidheight;
}

float Quectel_L80_GPS::getSpeed(void)
{
    return speed;
}

float Quectel_L80_GPS::getAngle(void)
{
    return angle;
}

float Quectel_L80_GPS::getMagVariation(void)
{
    return magvariation;
}

float Quectel_L80_GPS::getHDOP(void)
{
    return HDOP;
}

char Quectel_L80_GPS::getLatCardinalDir(void)
{
    return lat;
}

char Quectel_L80_GPS::getLonCardinalDir(void)
{
    return lon;
}

char Quectel_L80_GPS::getMagCardinalDir(void)
{
    return mag;
}

bool Quectel_L80_GPS::isFixed(void)
{
    return fix;
}

uint8_t Quectel_L80_GPS::getQuality(void)
{
    return fixQuality;
}

uint8_t Quectel_L80_GPS::getSatellites(void)
{
    return satellites;
}

bool Quectel_L80_GPS::LOCUS_StartLogger(void)
{
    sendCommand(PMTK_LOCUS_STARTLOG);
    recvdflag = false;
    return waitForSentence(PMTK_LOCUS_STARTSTOPACK);
}

bool Quectel_L80_GPS::LOCUS_StopLogger(void)
{
    sendCommand(PMTK_LOCUS_STOPLOG);
    recvdflag = false;
    return waitForSentence(PMTK_LOCUS_STARTSTOPACK);
}

bool Quectel_L80_GPS::LOCUS_ReadStatus(void)
{
    sendCommand(PMTK_LOCUS_QUERY_STATUS);

    if (!waitForSentence("$PMTKLOG"))
        return false;

    char *response = lastNMEA();
    uint16_t parsed[10];
    uint8_t i;

    for (i = 0; i < 10; i++) {
        parsed[i] = -1;
    }

    response = strchr(response, ',');
    for (i = 0; i < 10; i++) {
        if (!response || (response[0] == 0) || (response[0] == '*'))
            break;
        response++;
        parsed[i] = 0;
        while ((response[0] != ',') && (response[0] != '*') && (response[0] != 0)) {
            parsed[i] *= 10;
            char c = response[0];
            if (isdigit(c))
                parsed[i] += c - '0';
            else
                parsed[i] = c;
            response++;
        }
    }
    LOCUS_serial = parsed[0];
    LOCUS_type = parsed[1];
    if (isalpha(parsed[2])) {
        parsed[2] = parsed[2] - 'a' + 10;
    }
    LOCUS_mode = parsed[2];
    LOCUS_config = parsed[3];
    LOCUS_interval = parsed[4];
    LOCUS_distance = parsed[5];
    LOCUS_speed = parsed[6];
    LOCUS_status = !parsed[7];
    LOCUS_records = parsed[8];
    LOCUS_percent = parsed[9];

    return true;
}

uint16_t Quectel_L80_GPS::LOCUS_GetSerial()
{
    return LOCUS_serial;
}

uint16_t Quectel_L80_GPS::LOCUS_GetRecords()
{
    return LOCUS_records;
}

uint8_t Quectel_L80_GPS::LOCUS_GetType()
{
    return LOCUS_type;
}

uint8_t Quectel_L80_GPS::LOCUS_GetMode()
{
    return LOCUS_mode;
}

uint8_t Quectel_L80_GPS::LOCUS_GetConfig()
{
    return LOCUS_config;
}

uint8_t Quectel_L80_GPS::LOCUS_GetInterval()
{
    return LOCUS_interval;
}

uint8_t Quectel_L80_GPS::LOCUS_GetDistance()
{
    return LOCUS_distance;
}

uint8_t Quectel_L80_GPS::LOCUS_GetSpeed()
{
    return LOCUS_speed;
}

uint8_t Quectel_L80_GPS::LOCUS_GetStatus()
{
    return LOCUS_status;
}

uint8_t Quectel_L80_GPS::LOCUS_GetPercent()
{
    return LOCUS_percent;
}

bool Quectel_L80_GPS::setStandbyMode(void)
{
    if (inStandbyMode) {
        /* Returns false if already in standby mode */
        return false;
    } else {
        inStandbyMode = true;
        inFullOnMode = false;
        inAlwaysLocateMode = false;
        recvdflag = false;
        sendCommand(PMTK_STANDBY);
        return waitForSentence(PMTK_STANDBY_SUCCESS);
    }
}

bool Quectel_L80_GPS::wakeupStandby(void)
{
    if (inStandbyMode) {
        inStandbyMode = false;
        inFullOnMode = true;
        inAlwaysLocateMode = false;
        recvdflag = false;
        sendCommand("");  /* send byte to wake it up */
        return waitForSentence(PMTK_AWAKE);
    } else {
        /* Returns false if not in standby mode, nothing to wakeup */
        return false;
    }
}

/* AlwaysLocate Mode APIs: */
bool Quectel_L80_GPS::setAlwaysLocateMode(void)
{
    if (inAlwaysLocateMode) {
        /* Returns false if already in AlwayLocate mode,
         * so that you do not wake it up by sending commands to GPS */
        return false;
    } else {
        inStandbyMode = false;
        inFullOnMode = false;
        inAlwaysLocateMode = true;
        recvdflag = false;
        sendCommand(PMTK_ALWAYSLOCATE_STANDBY);
        return waitForSentence(PMTK_ALWAYSLOCATE_STANDBY_SUCCESS);
    }
}

bool Quectel_L80_GPS::wakeupAlwaysLocate(void)
{
    if (inAlwaysLocateMode) {
        inStandbyMode = false;
        inFullOnMode = true;
        inAlwaysLocateMode = false;
        sendCommand(PMTK_ALWAYSLOCATE_EXIT);
        return true;
    } else {
        /* Returns false if not in AlwaysLocate mode, nothing to wakeup */
        return false;
    }
}