#include "dot_util.h"
#include "RadioEvent.h"
#include "TSL2561.h"

 
#if ACTIVE_EXAMPLE == OTA_EXAMPLE

/////////////////////////////////////////////////////////////////////////////
// -------------------- DOT LIBRARY REQUIRED ------------------------------//
// * Because these example programs can be used for both mDot and xDot     //
//     devices, the LoRa stack is not included. The libmDot library should //
//     be imported if building for mDot devices. The libxDot library       //
//     should be imported if building for xDot devices.                    //
// * https://developer.mbed.org/teams/MultiTech/code/libmDot-dev-mbed5/    //
// * https://developer.mbed.org/teams/MultiTech/code/libmDot-mbed5/        //
// * https://developer.mbed.org/teams/MultiTech/code/libxDot-dev-mbed5/    //
// * https://developer.mbed.org/teams/MultiTech/code/libxDot-mbed5/        //
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////
// * these options must match the settings on your gateway //
// * edit their values to match your configuration         //
// * frequency sub band is only relevant for the 915 bands //
// * either the network name and passphrase can be used or //
//     the network ID (8 bytes) and KEY (16 bytes)         //
/////////////////////////////////////////////////////////////

// wifi gateway
static std::string network_name = "MTCDT-19400691";
static std::string network_passphrase = "MTCDT-19400691";

// cellular gateway
//static std::string network_name = "iuiot-gw1";
//static std::string network_passphrase = "pivotiot";


static uint8_t network_id[] = { 0x6C, 0x4E, 0xEF, 0x66, 0xF4, 0x79, 0x86, 0xA6 };
static uint8_t network_key[] = { 0x1F, 0x33, 0xA1, 0x70, 0xA5, 0xF1, 0xFD, 0xA0, 0xAB, 0x69, 0x7A, 0xAE, 0x2B, 0x95, 0x91, 0x6B };
static uint8_t frequency_sub_band = 1;
static lora::NetworkType public_network;// = lora::PUBLIC_LORAWAN;
static uint8_t join_delay = 1;
static uint8_t ack = 0;
static bool adr = true;

// deepsleep consumes slightly less current than sleep
// in sleep mode, IO state is maintained, RAM is retained, and application will resume after waking up
// in deepsleep mode, IOs float, RAM is lost, and application will start from beginning after waking up
// if deep_sleep == true, device will enter deepsleep mode
static bool deep_sleep = false;
static const uint16_t HOURLY_SLEEP_INTERVAL = 5; // 1 hour      *** changed for demo 1 min
static const uint16_t DAILY_SLEEP_INTERVAL = 43200; // 12 hours

mDot* dot = NULL;
lora::ChannelPlan* plan = NULL;

DigitalOut led1(LED1);
Serial pc(PA_9, PA_10);
TSL2561 lightSensor(PB_9, PB_8);

int main() {
  pc.baud(115200);
  pc.printf("main started \r\n");
  // Custom event handler for automatically displaying RX data
  RadioEvent events;
    

  /////////////////////////////
  // visual display
  led1 = 0;
  for(int i = 0; i < 5; i++) {
    led1 = 1;
    wait(1);
    led1 = 0;
    wait(1);
  }

  // set gain on light sensor
  lightSensor.set_timing_reg(TIMING_GAIN_16);

  
  mts::MTSLog::setLogLevel(mts::MTSLog::TRACE_LEVEL);
  
#if CHANNEL_PLAN == CP_US915
  plan = new lora::ChannelPlan_US915();
#elif CHANNEL_PLAN == CP_AU915
  plan = new lora::ChannelPlan_AU915();
#elif CHANNEL_PLAN == CP_EU868
  plan = new lora::ChannelPlan_EU868();
#elif CHANNEL_PLAN == CP_KR920
  plan = new lora::ChannelPlan_KR920();
#elif CHANNEL_PLAN == CP_AS923
  plan = new lora::ChannelPlan_AS923();
#elif CHANNEL_PLAN == CP_AS923_JAPAN
  plan = new lora::ChannelPlan_AS923_Japan();
#elif CHANNEL_PLAN == CP_IN865
  plan = new lora::ChannelPlan_IN865();
#endif
  assert(plan);
  
  dot = mDot::getInstance(plan);
  assert(dot);


  // attach the custom events handler
  dot->setEvents(&events);
  
  if (!dot->getStandbyFlag()) {
    logInfo("mbed-os library version: %d", MBED_LIBRARY_VERSION);
    
    // start from a well-known state
    logInfo("defaulting Dot configuration");
    dot->resetConfig();
    dot->resetNetworkSession();

    // set the data_rate
    dot->setTxDataRate(lora::DR_1);
    
    // make sure library logging is turned on
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);
    
    // update configuration if necessary
    if (dot->getJoinMode() != mDot::OTA) {
      logInfo("changing network join mode to OTA");
      if (dot->setJoinMode(mDot::OTA) != mDot::MDOT_OK) {
	logError("failed to set network join mode to OTA");
      }
    }
    // in OTA and AUTO_OTA join modes, the credentials can be passed to the library as a name and passphrase or an ID and KEY
    // only one method or the other should be used!
    // network ID = crc64(network name)
    // network KEY = cmac(network passphrase)
    update_ota_config_name_phrase(network_name, network_passphrase, frequency_sub_band, public_network, ack);
    //update_ota_config_id_key(network_id, network_key, frequency_sub_band, public_network, ack); 
    
    // configure network link checks
    // network link checks are a good alternative to requiring the gateway to ACK every packet and should allow a single gateway to handle more Dots
    // check the link every count packets
    // declare the Dot disconnected after threshold failed link checks
    // for count = 3 and threshold = 5, the Dot will ask for a link check response every 5 packets and will consider the connection lost if it fails to receive 3 responses in a row
    update_network_link_check_config(3, 5);
    
    // enable or disable Adaptive Data Rate
    dot->setAdr(adr);
    
    // Configure the join delay
    dot->setJoinDelay(join_delay);
    
    pc.printf("network information set\r\n");

    // save changes to configuration
    logInfo("saving configuration");
    if (!dot->saveConfig()) {
      logError("failed to save configuration");
    }
    
    // display configuration
    display_config();
  } else {
    // restore the saved session if the dot woke from deepsleep mode
    // useful to use with deepsleep because session info is otherwise lost when the dot enters deepsleep
    logInfo("restoring network session from NVM");
    dot->restoreNetworkSession();
  }
  
  uint8_t counter = 0;
  std::vector<uint8_t> tx_data;
  char buffer[5];

  while (true) {
    
    // join network if not joined
    if (!dot->getNetworkJoinStatus()) {
      pc.printf("joining lora network.\r\n");
      join_network(&pc);
    } 

    
    // hourly loop
    while( counter < 12) {

      uint8_t light_data = get_light_data(&lightSensor, &pc);
      //pc.printf("pseudo-light data: %d\r\n", light_data);
      pc.printf("\r\nhour: %d\r\n", ++counter);
      
      
      // clean out char buffer
      strcpy(buffer, "XXXX");

      // convert to ascii char && push to vector
      ascii_converter(buffer, light_data, &tx_data, &pc);
      pc.printf("char buffer: %s\r\n", buffer);
      
      if( counter < 12)
	tx_data.push_back(' ');
      

      // if going into deepsleep mode, save the session so we don't need to join again after waking up
      // not necessary if going into sleep mode since RAM is retained
      if (deep_sleep) {
	logInfo("saving network session to NVM");
	dot->saveNetworkSession();
      }
      
      // ONLY ONE of the three functions below should be uncommented depending on the desired wakeup method
      sleep_wake_rtc_only(deep_sleep, HOURLY_SLEEP_INTERVAL, &pc);
      //sleep_wake_interrupt_only(deep_sleep);
      //sleep_wake_rtc_or_interrupt(deep_sleep);
    }

    //print vector data
    pc.printf("vector data:");
    
    for(std::vector<uint8_t>::const_iterator iter = tx_data.begin(); iter != tx_data.end(); ++iter) {
      pc.printf(" %c",*iter);
    }
    pc.printf("\r\n");

    pc.printf("capacity tx_data: %d\r\n", tx_data.capacity());
    pc.printf("sized tx_data: %d\r\n", tx_data.size());
    //actually send the data
    pc.printf("sending data\r\n");
    

    // set the data_rate (from forum)
    dot->setTxDataRate(lora::DR_1);
    int retVal = send_data(tx_data, &pc);
    //while(retVal == 1)
    //retVal = send_data(tx_data, &pc);

    // erase vector data & reset counter
    tx_data.erase(tx_data.begin(), tx_data.end());
    counter = 0;
    pc.printf("deleted tx_data and reset counter\r\ncap: %d size: %d\r\n", tx_data.capacity(), tx_data.size());

    // sleep for the day  *** don't want to sleep for 12 hours for the demo
    //sleep_wake_rtc_only(deep_sleep, DAILY_SLEEP_INTERVAL, &pc);
    
  }
  
  return 0;
}

#endif
