// Simple game of Doodle Jump with a BMP image, sound, and IMU control.
#include "mbed.h"
#include "rtos.h"
#include "SDFileSystem.h" /* SD card library */
#include "uLCD_4DGL.h" /* LCD library */
#include "LSM9DS1.h" /* IMU library */
#include "wave_player.h" /* wav file playing */
#include "myBMP.h" /* BMP image handling */
#include <stdlib.h>     /* srand, rand */
#include <time.h>       /* time */

//#define PI 3.14159
//#define DECLINATION -4.94 // Declination (degrees) in Atlanta,GA.
#define GRAVITY 3 /* acceleration per frame */
#define MAX_PLATFORMS 5
#define MAX_ENEMIES 2
#define PLAYER_WIDTH 9
#define PLAYER_HEIGHT 10
// range of y values platforms can be between
#define MIN_PLATFORM_INTERVAL 15
#define MAX_PLATFORM_INTERVAL 40
#define PLATFORM_WIDTH 20
#define PLATFORM_HEIGHT 5
#define INIT_PLAYER_VELOCITY -15


uLCD_4DGL uLCD(p28,p27,p30); // serial tx, serial rx, reset pin
LSM9DS1 IMU(p9, p10, 0xD6, 0x3C);
SDFileSystem sd(p5, p6, p7, p8, "sd");
AnalogOut DACout(p18); //analog out for speaker
wave_player waver(&DACout);     

DigitalOut myled1(LED1);
DigitalOut myled2(LED2);               

Mutex sound_mutex;
Thread * t1 = NULL; // must set to null for initial delete, calling delete on a null pointer is safe

typedef struct xycoord {
    int x;
    int y;
} xycoord; 

xycoord platforms[MAX_PLATFORMS];

// play a song of given filename, void const * input for use with threads
// thread safe, will not try to play more than one sound at once
void playSound(void const * args);

// return index of platform with lowest y coordinates (highest on screen)
int getHighestPlatform();

// return index of platform with highest y coordinates (lowst on screen)
int getLowestPlatform();

// if a generate x, y coordinates above highest platform
void getNextPlatform(int &x, int &y);

// create all platforms
void initPlatforms();

// try to play a sound, do nothing if mutex locked
void trySound(const char * file);
    
int main()
{
    wait(2.0f); // wait a bit for initialization
    uLCD.cls();
    // jack up LCD baud rate to max for fast display
    uLCD.baudrate(3000000); 
    trySound("/sd/DMX.wav"); // play init sound
  
    // set up IMU
    uLCD.printf("Calibrating IMU, place on flat surface. Please wait...");
    IMU.begin();
    if (!IMU.begin()) {
        uLCD.printf("Failed to communicate with LSM9DS1.\n");
        wait(5.0f);
        return -1;
    }
    IMU.calibrate(1);
    IMU.calibrateMag(0);
    
    //start countdown timer for player to start playing
    
    uLCD.cls();
    uLCD.locate(0,8);
    uLCD.printf("Starting in: 3");
    wait(1.0f);
    uLCD.cls();
    uLCD.locate(0,8);
    uLCD.printf("Starting in: 2");
    wait(1.0f);
    uLCD.cls();
    uLCD.locate(0,8);
    uLCD.printf("Starting in: 1");
    wait(0.6f);
    uLCD.cls();
    uLCD.locate(0,15);
    uLCD.printf("TA SAYS WHAT");
    
    
    
    //float fx=63.0,fy=63.0,vx=1.0,vy=0.4;
    // x, y is player x and y, player_velocity is player movement in pixels/frame, absolute_y is the y pixels the camera has "moved" up, old_x old_y for drawing over old player image
    int x = 64+(PLAYER_WIDTH + 1)/2, y = 127-PLAYER_HEIGHT; //,radius=4;
    int player_velocity = INIT_PLAYER_VELOCITY;
    int absolute_y = 0;
    int old_x = x, old_y = y;
    uLCD.background_color(BLACK);
    
    
    
    float ax; // accelerometer reading
    
    RGBApixel *Colors = new RGBApixel [2];
    
    // init random number generator, remove text from IMU
    srand(time(0));
    uLCD.cls();
    
    // initialize platforms and prepare them for drawing
    initPlatforms();
    int lowestPlatformIndex;
    int platx;
    int platy;
    
    // store the pixel values of player in memory to reduce the drawing time per frame
    // note that this image should be small, mbed's memory is not very large
    int * color_array = new int[PLAYER_WIDTH*PLAYER_HEIGHT];
    ReadColorsFromFile("/sd/stardesu.bmp", Colors, color_array, x, y);
    
    // make score color white
    uLCD.color(WHITE);
    
    // enter main game loop
    while (1) 
    {
        // read from IMU
        while(!IMU.accelAvailable());
        IMU.readAccel();
        ax = IMU.calcAccel(IMU.ax);
        
        // get lowest platform
        lowestPlatformIndex = getLowestPlatform();
        platx = platforms[lowestPlatformIndex].x;
        platy = platforms[lowestPlatformIndex].y;
        
        // platform is off screen, create a new one
        if (platy > 127)
        {
            getNextPlatform(platx, platy);
            platforms[lowestPlatformIndex].y = platy;
            platforms[lowestPlatformIndex].x = platx;
        }
        
        // check collisions of player with platforms
        for (int i = 0; i < MAX_PLATFORMS; i++)
        {
            if (player_velocity > 0 && (x+PLAYER_WIDTH) > platforms[i].x && x < (platforms[i].x + PLATFORM_WIDTH) && (y+PLAYER_HEIGHT) >= (platforms[i].y) && (y+PLAYER_HEIGHT) <= (platforms[i].y + PLATFORM_HEIGHT+7))
            {
                 player_velocity = INIT_PLAYER_VELOCITY;   
                 y = platforms[i].y-(PLAYER_HEIGHT-1);
                 // bounce sound
                 trySound("/sd/DMX3.wav");
                 break;
            }
        }
        
        
        
        //erase old player location by drawing over it
        uLCD.filled_rectangle(old_x, old_y, old_x+PLAYER_WIDTH, old_y+PLAYER_HEIGHT, BLACK);
        
        // if player is at middle of screen, stop moving player and start moving platforms relative to player
        // "halfway point" is at pixal 64 = 63, force player to be unable to go above 63
        if (y < 63)
        {
               int difference = 63 - y;
               y = 63;
               absolute_y+=difference;
               for (int i = 0; i < MAX_PLATFORMS; i++)
               {
                   platx = platforms[i].x;
                   platy = platforms[i].y;
                   // draw over old location
                   uLCD.filled_rectangle(platx, platy, platx+PLATFORM_WIDTH, platy+PLATFORM_HEIGHT, BLACK);
                   platforms[i].y+=difference;
               }           
        }
        // draw platforms
        for (int i = 0; i < MAX_PLATFORMS; i++)
        {
            uLCD.filled_rectangle(platforms[i].x, platforms[i].y, platforms[i].x + PLATFORM_WIDTH - 1, platforms[i].y + PLATFORM_HEIGHT - 1, GREEN);
        }
        // draw new player location
        //ReadBMPFromFile("/sd/stardesu.bmp", Colors, &uLCD, x,y);
        DrawColorstoLCD(color_array, &uLCD, x, y, PLAYER_WIDTH, PLAYER_HEIGHT);
        wait(.05f);
        // print absolute y as player score
        uLCD.locate(0, 0);
        uLCD.printf("Score: %i", absolute_y);
        
        // save previous player coordinates, update player coordinates
        old_x = x;
        old_y = y;
        x = 64-PLAYER_WIDTH/2 + (int)(64.0f * ax);
        y += player_velocity;
        // apply "gravity" and "terminal velocity" to current player velocity
        player_velocity += GRAVITY;
        if (player_velocity > -1*INIT_PLAYER_VELOCITY)
        {
            player_velocity = -1*INIT_PLAYER_VELOCITY;
        }
        
        
        
        
        
        
        // check game over condition
        if ( y >= 127-PLAYER_HEIGHT && player_velocity > 0)
        {
            // kill player, game over
            wait(.1f);
            trySound("/sd/RR.wav");
            trySound("/sd/RR.wav");
            wait(.1f);
            while(1)
            {
                uLCD.cls();
                for (int i = 0; i < 20; i++)
                {
                    uLCD.color(RED);
                    uLCD.printf("GET REKT");
                    uLCD.color(GREEN);
                    uLCD.printf("GET REKT");
                }
                wait(.1f);
                uLCD.cls();
                for (int i = 0; i < 80; i++)
                {
                    uLCD.color(RED);
                    uLCD.printf("(=");
                    uLCD.color(GREEN);
                    uLCD.printf("(=");
                }
                wait(.1f);
            }
            //player_velocity = INIT_PLAYER_VELOCITY;
            //y = 127-PLAYER_HEIGHT;
        } 
    } // end of main loop
} // end of main

// function implementations

void playSound(void const * args) 
{
    sound_mutex.lock();
    FILE *wave_file;
    char const * actual_args = (char const *)args;
    wave_file = fopen(actual_args,"r"); // change to whatever sound
    if (wave_file == NULL)
    {
        myled1 = 1;
    }
    else 
    {
        myled2 = 1;
    }
    waver.play(wave_file);
    fclose(wave_file);
    sound_mutex.unlock();
}


int getHighestPlatform()
{
    int highest_index = 0;
    for (int i = 1; i < MAX_PLATFORMS; i++)
    {  
        if (platforms[i].y < platforms[highest_index].y)
            highest_index = i;
    }
    return highest_index;
}


int getLowestPlatform()
{
    int lowest_index = 0;
    for (int i = 1; i < MAX_PLATFORMS; i++)
    {  
        if (platforms[i].y > platforms[lowest_index].y)
            lowest_index = i;
    }
    return lowest_index;
}


void getNextPlatform(int &x, int &y)
{  
    x = rand()%(128-PLATFORM_WIDTH+1);
    y = platforms[getHighestPlatform()].y - (rand()%(MAX_PLATFORM_INTERVAL-MIN_PLATFORM_INTERVAL+1) + MIN_PLATFORM_INTERVAL);
}


void initPlatforms()
{
    // first set all to 0
    for (int i = 0; i < MAX_PLATFORMS; i++)
    {
        platforms[i].x = 0;
        platforms[i].y = 127;
    }
    int x;
    int y;
    for (int i = 0; i < MAX_PLATFORMS; i++)
    {
        getNextPlatform(x, y);
        platforms[i].x = x;
        platforms[i].y = y;
    }
}


void trySound(const char * file)
{
    if (sound_mutex.trylock())
    {   
        delete t1;
        t1 = new Thread(playSound, (void*)file);
        sound_mutex.unlock();
    }
}
