/*****************************************************************************/
/*      Main program for the implementation of RTOS_ALARM Clock              */
/*Function: To sound an alarm when the actual time is equal to alarm time    */
/*Features: USB access to set the time                                       */
/*          User can set the time with the joystick                          */
/*          Audio Visual alarm available                                     */
/*          Alarm turns OFF automaticaaly after 1 min if not turned OFF      */
/*Author:   Bhakti Kulkarni                                                  */
/*Date:     03/25/2014                                                       */
/*****************************************************************************/

#include "mbed.h"
#include "C12832_lcd.h"
#include "DebouncedIn.h"
#include "DebouncedInterrupt.h"
#include "rtos.h"
#include "USBSerial.h"

#define INTERRUPT
#define HR 1
#define MIN 0
#define DURATION 1
#define MAX_HRS 24
#define MAX_MINS 60
#define MAX_SECS 60

C12832_LCD lcd;
USBSerial serial;

DebouncedInterrupt increase(p12);                     //Switch to increase time
DebouncedInterrupt select(p14);         //Switch to toggle between hrs and mins
Mutex M_lcd;                            //Mutex for LCD
DebouncedIn stop_alarm(p16);            //Switch to stop the alarm
Ticker update_time;                     //Ticker to update actual time
DigitalOut wake_up(LED2);               //Visual alarm LED
PwmOut speaker(p26);                    //Speaker for audible alarm

/*Global variables for mins and hrs*/
int hrs;
int mins;
int hrs_set = MAX_HRS;
int mins_set = MAX_MINS;
int sec_set = 0;
bool selection= HR;            //Signal to indicate the hrs/mins to be changed

/*****************************************************************************/
/*                     ISR for increasing the time                           */
/*Function: This routine checks the value of selection signal.               */
/*          If selection value - HR it updates hrs and if selection = MIN    */
/*          it updates the minute value. It also checks for the maximum      */
/*          value of hrs and mins and reset appropriate variable             */
/*****************************************************************************/
void increase_time()
{
    if (selection == HR)
    { 
      if (hrs < MAX_HRS)
         hrs ++;
      else
         hrs = 0;
    }
    else
    {
        if (mins < MAX_MINS)
           mins ++;
         else{
           mins = 0;
           if (hrs < MAX_HRS)
            hrs++;
           else
             hrs = 0;
        }
     }
}
/*****************************************************************************/

/*****************************************************************************/
/*                          ISR to update selection                          */
/*Function: Allows the user to toggle between the hours and minutes          */
/*****************************************************************************/
void select_time()
{
    selection = !selection;
}
/*****************************************************************************/

/*****************************************************************************/
/*                        Thread to update LCD                               */
/*Function: This thread updates the LCD with the actual and the alarm time   */
/*****************************************************************************/
void update_lcd(void const *args)
{
    M_lcd.lock();                                       //Mutex to lock the LCD
    lcd.cls();
    lcd.locate(0,20);
    while(1) {
        lcd.cls();
        lcd.locate(0,0);
        lcd.printf("Actual-hh:mm:sec %02d:%02d:%02d",hrs_set,mins_set,sec_set);
        lcd.locate(0,10);
        lcd.printf("Alarm-hh:mm:sec %02d:%02d:%02d",hrs,mins,0);
        M_lcd.unlock();  
        Thread::wait(25);
    }
}
/*****************************************************************************/

/*****************************************************************************/
/*                   Ticker function to update the actual time               */
/*This function is called every second. It increments no. of seconds         */
/*Also checks for max secs, mins and hours and accordingly adjusts other     */
/*variables.                                                                 */
/*****************************************************************************/
void time_update()
{
    sec_set++;
    if (sec_set == MAX_MINS)
    {
        mins_set++;
        sec_set = 0;
    }
    if (mins_set == MAX_MINS)
    {
        hrs_set++;
        mins_set = 0;
    }
    if (hrs_set == MAX_HRS)
    {
        hrs_set = 0;
        mins_set = 0;
        sec_set = 0;
    }
}
/*****************************************************************************/

/*****************************************************************************/
/*                        Function to sound the alarm                        */
/*This function is called when the alarm time = actual time                  */
/*It sounds the alarm till alarm_stop button is pressed | 1 min is ellapsed  */
/*The variable frequency and beat rate of the PWM decide the tone of alarm   */
/*****************************************************************************/
void alarm_on()
{
    float frequency[]={659,554,659,554,440,494,554,587,494,659,554,440};
    float beat[]={1,1,1,1,1,0.5,0.5,1,1,1,1,2}; //beat array
    while (!stop_alarm && !(hrs_set == hrs && mins_set == mins+DURATION)) {
        wake_up = 1;
        for (int i=0;i<=11;i++) {
            speaker.period(1/(2*frequency[i])); // set PWM period
            speaker=0.5; // set duty cycle
            wait(0.4*beat[i]); // hold for beat period
        }
    }
    wake_up =0;                                       //Turn OFF visual alarm
    speaker = 0.0;                                     //Turn OFF audio alarm
}
/*****************************************************************************/

/*****************************************************************************/
/*                    Thread for time Comparison                             */
/*This thread comapres the actual and the alarm time and calls the alarm     */
/*function when both the timings are equal. It also reset the alarm time     */
/*after the alarm is turned OFF(either by user or after a minute is ellapsed)*/
/*****************************************************************************/
void sound_alarm(void const *args)
{
    while (true){
    if ((hrs == hrs_set && mins_set == mins && sec_set == 0) )
    {
        alarm_on();
        hrs = 0;
        mins = 0;
    }
    Thread::wait(50);
    }
}
/*****************************************************************************/

/*****************************************************************************/
/*                               Main Thread                                 */
/*Initializes all the threads and the USB inerface.                          */
/*This thread makes a blocking call to set the time. In this application     */
/*blocking call is used since the actual time is necessary parameter for the */
/*alarm clock. Without the time alarm clock has no function to do.           */
/*****************************************************************************/
int main() {
    char ch;
    increase.attach(&increase_time);
    select.attach(&select_time);
    Thread compare_time (sound_alarm,NULL,osPriorityAboveNormal);
    Thread lcd_display(update_lcd,NULL,  osPriorityAboveNormal);
    serial.printf("\r\n");
    serial.scanf("%c",&ch);
    serial.printf("Set the hours\r\n");
    do{
        serial.scanf("%d",&hrs_set);
        if (hrs_set > MAX_HRS)
            serial.printf("Enter Valid hours\r\n");
    }while (hrs_set > MAX_HRS);              //error checking for hours entered
    serial.printf("Set mins\r\n");
    do
    {
        serial.scanf("%d",&mins_set);
        if (mins_set > MAX_MINS)
            serial.printf("Enter Valid minutes\r\n");
    }while (mins_set > MAX_MINS);          //error checking for minutes entered
    update_time.attach(&time_update,1);
    while(1)
    {
        Thread::wait(1000);
    }
}
