/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <string.h>
#include "mbed.h"
#include "BMP180.h"
#include "Si7020.h"

// User I/O objects
DigitalOut red(LED_RED, LED_OFF);

// Sensor objects
I2C i2c(I2C_SDA, I2C_SCL);
AnalogIn light_ain(AIN_0D);
DigitalOut light_en(P0_2, 1);

BMP180 bmp180(&i2c);
Si7020 si7020(&i2c);

int currentPressure;
float currentHumidity;
float currentTemperature;
float currentLight;

// *****************************************************************************
int main(void)
{
    printf("\n\nEnvironmental sensor demo\n");

    /* Setup BMP180 */
    while(bmp180.init() != 0) {
        printf("Failed to initialize barometer\n");
        wait(1);
    }

    while (true) {

        // Start non-blocking readings
        bmp180.startPressure(BMP180::STANDARD);
        si7020.startHumidity();

        // Toggle LED and wait between readings
        red = 1;
        wait(1);
        red = 0;
    
        // Enable light sensor to charge capacitor
        light_en = 0;
        
        // Get Pressure
        if(bmp180.getPressure(&currentPressure) != 0) {
            printf("Error getting pressure\n");
        }

        // Get Humidity
        if(si7020.checkHumidity(&currentHumidity) != 0) {
            printf("Error getting humidity\n");
        }

        // Get temperature
        if(si7020.getPrevTemperature(&currentTemperature) != 0) {
            printf("Error getting temperature\n");
        }

        // Get light 
        currentLight = light_ain.read();
        
        // Disable light sensor
        light_en = 1;
        
        printf("Press = %0.1f kPa\n", (float)(currentPressure/1000.0));
        printf("Humid = %0.1f %%\n", currentHumidity);
        printf("Temp = %0.1f C\n", currentTemperature);
        printf("Light = %0.1f %% \n", currentLight*100.0f);
        printf("\n");
    }
}
