#ifndef __TIME_H_
#define __TIME_H_

#include <time.h>
#include "stdio.h"
#include "string.h"
#include <list>

class Time;

/**
    This class encapsulates a point in time - which means it will not change after it has been created.
*/
class TimeStamp {
    friend class Time;
public:
    /**
        @returns the year of the time stamp
    */
    int getYear() {
        return _tm.tm_year+1900;
    };

    /**
        @returns the month of the time stamp
    */
    int getMonth() {
        return _tm.tm_mon;
    };

    int getDay() {
    /**
        @returns the day-of-the-month of the time stamp
    */
        return _tm.tm_mday;
    };

    /**
        @returns the hour of the time stamp
    */
    int getHour() {
        return _tm.tm_hour;
    };

    /**
        @returns the minute of the time stamp
    */
    int getMinute() {
        return _tm.tm_min;
    };

    /**
        @returns the second of the time stamp
    */
    int getSecond() {
        return _tm.tm_sec;
    };

    /**
        get the day of the week - monday is 0
        @returns the day-of-the-week of the time stamp
    */
    int getDayOfWeek() {
        int dow=_tm.tm_wday;
        dow--;
        if (dow==-1)
            dow=6;
        return dow;
    };

    /**
        @returns the day-of-the-year of the time stamp
    */
    int getDayOfYear() {
        return _tm.tm_yday;
    };

    /**
        creates a new time stamp based on the UNIX time stamp (seconds since 1970)
    */
    TimeStamp(time_t unixTime) {
        updateTime(unixTime);
    };

    ~TimeStamp() {
    };
    
    /**
        @param ts the time stamp to compare to
        @returns true when both timestamp are the same point in time
    */
    bool isSame(TimeStamp* ts);
    
    /**
        @param ts the time stamp to compare to
        @return true when the current time stamp is before the given time stamp
    */
    bool isBefore(TimeStamp* ts);
    /**
        @param ts the time stamp to compare to
        @return true when the current time stamp is after the given time stamp
    */
    bool isAfter(TimeStamp* ts);
private:
    void updateTime(time_t unixTime) {
        struct tm *time;
        time=localtime(&unixTime);
        memcpy(&_tm, time, sizeof(_tm));
    }
    TimeStamp(int year, int mon, int day, int hour, int min, int sec)
    {
        _tm.tm_year=year-1900;
        _tm.tm_mon=mon;
        _tm.tm_mday=day;
        _tm.tm_hour=hour;
        _tm.tm_min=min;
        _tm.tm_sec=sec;
    }

    struct tm _tm;
};

class TimeZoneEntry;

/**
    This class handles the time zone calculation, and is used for creating time stamp objects.
*/
class Time {
public:
    /**
        creates a new Time instance. On the first call, it reads the file 'timezone.csv' from the USB disk (local file system).
    */
    Time();
    ~Time();
    /**
        creates a new TimeStamp instance. The caller is responsible for deleting it afterwards!
        @returns the time stamp
    */
    TimeStamp* getTime();
    /**
        @returns the current time stamp (UTC) in UNIX format, without time zone correction.
    */
    long getUnixTime();
private:
    static std::list<TimeZoneEntry*> *_timeZoneEntries;
    void readTimeZones();
};

#endif