/**
 * Note: This is a cross-platform version which is separated from hardware.
 * ---------------------------------------------------------------------------
 * This module is modified by Chun-Feng Huang for abstracting and more functionality.
 * Modified by: Chun-Feng Huang
 * E-mail: bens0516@gmail.com
 *
*/

//----------------------------------//
/**
 * @author Aaron Berk
 *
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Quadrature Encoder Interface.
 *
 * A quadrature encoder consists of two code tracks on a disc which are 90
 * degrees out of phase. It can be used to determine how far a wheel has
 * rotated, relative to a known starting position.
 *
 * Only one code track changes at a time leading to a more robust system than
 * a single track, because any jitter around any edge won't cause a state
 * change as the other track will remain constant.
 *
 * Encoders can be a homebrew affair, consisting of infrared emitters/receivers
 * and paper code tracks consisting of alternating black and white sections;
 * alternatively, complete disk and PCB emitter/receiver encoder systems can
 * be bought, but the interface, regardless of implementation is the same.
 *
 *               +-----+     +-----+     +-----+
 * Channel A     |  ^  |     |     |     |     |
 *            ---+  ^  +-----+     +-----+     +-----
 *               ^  ^
 *               ^  +-----+     +-----+     +-----+
 * Channel B     ^  |     |     |     |     |     |
 *            ------+     +-----+     +-----+     +-----
 *               ^  ^
 *               ^  ^
 *               90deg
 *
 * The interface uses X2 encoding by default which calculates the pulse count
 * based on reading the current state after each rising and falling edge of
 * channel A.
 *
 *               +-----+     +-----+     +-----+
 * Channel A     |     |     |     |     |     |
 *            ---+     +-----+     +-----+     +-----
 *               ^     ^     ^     ^     ^
 *               ^  +-----+  ^  +-----+  ^  +-----+
 * Channel B     ^  |  ^  |  ^  |  ^  |  ^  |     |
 *            ------+  ^  +-----+  ^  +-----+     +--
 *               ^     ^     ^     ^     ^
 *               ^     ^     ^     ^     ^
 * Pulse count 0 1     2     3     4     5  ...
 *
 * This interface can also use X4 encoding which calculates the pulse count
 * based on reading the current state after each rising and falling edge of
 * either channel.
 *
 *               +-----+     +-----+     +-----+
 * Channel A     |     |     |     |     |     |
 *            ---+     +-----+     +-----+     +-----
 *               ^     ^     ^     ^     ^
 *               ^  +-----+  ^  +-----+  ^  +-----+
 * Channel B     ^  |  ^  |  ^  |  ^  |  ^  |     |
 *            ------+  ^  +-----+  ^  +-----+     +--
 *               ^  ^  ^  ^  ^  ^  ^  ^  ^  ^
 *               ^  ^  ^  ^  ^  ^  ^  ^  ^  ^
 * Pulse count 0 1  2  3  4  5  6  7  8  9  ...
 *
 * It defaults
 *
 * An optional index channel can be used which determines when a full
 * revolution has occured.
 *
 * If a 4 pules per revolution encoder was used, with X4 encoding,
 * the following would be observed.
 *
 *               +-----+     +-----+     +-----+
 * Channel A     |     |     |     |     |     |
 *            ---+     +-----+     +-----+     +-----
 *               ^     ^     ^     ^     ^
 *               ^  +-----+  ^  +-----+  ^  +-----+
 * Channel B     ^  |  ^  |  ^  |  ^  |  ^  |     |
 *            ------+  ^  +-----+  ^  +-----+     +--
 *               ^  ^  ^  ^  ^  ^  ^  ^  ^  ^
 *               ^  ^  ^  ^  ^  ^  ^  ^  ^  ^
 *               ^  ^  ^  +--+  ^  ^  +--+  ^
 *               ^  ^  ^  |  |  ^  ^  |  |  ^
 * Index      ------------+  +--------+  +-----------
 *               ^  ^  ^  ^  ^  ^  ^  ^  ^  ^
 * Pulse count 0 1  2  3  4  5  6  7  8  9  ...
 * Rev.  count 0          1           2
 *
 * Rotational position in degrees can be calculated by:
 *
 * (pulse count / X * N) * 360
 *
 * Where X is the encoding type [e.g. X4 encoding => X=4], and N is the number
 * of pulses per revolution.
 *
 * Linear position can be calculated by:
 *
 * (pulse count / X * N) * (1 / PPI)
 *
 * Where X is encoding type [e.g. X4 encoding => X=44], N is the number of
 * pulses per revolution, and PPI is pulses per inch, or the equivalent for
 * any other unit of displacement. PPI can be calculated by taking the
 * circumference of the wheel or encoder disk and dividing it by the number
 * of pulses per revolution.
 */

/**
 * Includes
 */
#include "ENCODER_PROCESSOR.h"

ENCODER_PROCESSOR::ENCODER_PROCESSOR(int pulsesPerRevolution_in, int size_MA_window_in, double sampling_time_in, Encoding encoding):
        pulsesPerRevolution(pulsesPerRevolution_in), size_MA_window(size_MA_window_in), Ts(sampling_time_in)
{
    //
    is_initiated = false;


    //
    MA_window.assign(size_MA_window, 0);

    delta_count = 0;
    idx_MA_array = 0;


    pulses_       = 0;
    revolutions_  = 0;
    encoding_     = encoding;

    // Findout what the current state is.
    // int chanA = channelA_.read();
    // int chanB = channelB_.read();

    //2-bit state.
    encoderState_now = 0; // (chanA << 1) | (chanB);
    encoderState_pre = encoderState_now;

    // Attaching call back function
    //X2 encoding uses interrupts on only channel A.
    //X4 encoding uses interrupts on      channel A and on channel B.


    // Unit transformation
    // Rotational speed
    count_2_rad_s = 1.0/(pulsesPerRevolution*1.0)*6.2831852/Ts/(size_MA_window*1.0);
    count_2_deg_s = 1.0/(pulsesPerRevolution*1.0)*360.0/Ts/(size_MA_window*1.0);
    // Angle
    count_2_rad = 1.0/(pulsesPerRevolution*1.0)*6.2831852;
    count_2_deg = 1.0/(pulsesPerRevolution*1.0)*360.0;
}

// Process control
void ENCODER_PROCESSOR::reset(void) {

    pulses_      = 0;
    revolutions_ = 0;

}

// Encode (call Back)
/////////////////////////
// +-------------+
// | X2 Encoding |
// +-------------+
//
// When observing states two patterns will appear:
//
// Counter clockwise rotation:
//
// 10 -> 01 -> 10 -> 01 -> ...
//
// Clockwise rotation:
//
// 11 -> 00 -> 11 -> 00 -> ...
//
// We consider counter clockwise rotation to be "forward" and
// counter clockwise to be "backward". Therefore pulse count will increase
// during counter clockwise rotation and decrease during clockwise rotation.
//
// +-------------+
// | X4 Encoding |
// +-------------+
//
// There are four possible states for a quadrature encoder which correspond to
// 2-bit gray code.
//
// A state change is only valid if of only one bit has changed.
// A state change is invalid if both bits have changed.
//
// Clockwise Rotation ->
//  (The bits are constructed as "AB")
//    00 01 11 10 00
//
// <- Counter Clockwise Rotation
//
// If we observe any valid state changes going from left to right, we have
// moved one pulse clockwise [we will consider this "backward" or "negative"].
//
// If we observe any valid state changes going from right to left we have
// moved one pulse counter clockwise [we will consider this "forward" or
// "positive"].
//
// We might enter an invalid state for a number of reasons which are hard to
// predict - if this is the case, it is generally safe to ignore it, update
// the state and carry on, with the error correcting itself shortly after.


//---------------------------------------------//
// Call-back function for both pin A and pin B interupt (rise and fall)
void ENCODER_PROCESSOR::IntrCB_pulseUpdate(int phase_A, int phase_B) {

    if (!is_initiated){
        //2-bit state.
        encoderState_now = (phase_A << 1) | (phase_B); // AB
        encoderState_pre = encoderState_now;
        //
        is_initiated = true;
        //
        return;
    }

    //--------------------------------//
    int change = 0;

    //2-bit state.
    encoderState_now = (phase_A << 1) | (phase_B); // AB

    if (encoding_ == X2_ENCODING) {

        //11->00->11->00 is counter clockwise rotation or "forward".
        if ((encoderState_pre == 0x3 && encoderState_now == 0x0) ||
                (encoderState_pre == 0x0 && encoderState_now == 0x3)) {

            pulses_++;

        }
        //10->01->10->01 is clockwise rotation or "backward".
        else if ((encoderState_pre == 0x2 && encoderState_now == 0x1) ||
                 (encoderState_pre == 0x1 && encoderState_now == 0x2)) {

            pulses_--;

        }

    } else{  // if (encoding_ == X4_ENCODING) {

        //Entered a new valid state.
        if (((encoderState_now ^ encoderState_pre) != INVALID) && (encoderState_now != encoderState_pre)) {
            // 2 bit state.
            // (Right hand bit of prev) XOR (left hand bit of current)
            // B_(n-1) xor A_n (because of the 90deg phase shift and A is leading B when rotates clockwise)
            // gives 0 if clockwise rotation and 1 if counter clockwise rotation.
            change = (encoderState_pre & PREV_MASK) ^ ((encoderState_now & CURR_MASK) >> 1);

            if (change == 0) {
                change = -1;
            }

            pulses_ -= change;
        }

    }

    encoderState_pre = encoderState_now;

}
// (Un-necessary) Call-back function for index-pin interupt
void ENCODER_PROCESSOR::IntrCB_indexUpdate(void) {
    revolutions_++;
}
//---------------------------------------------//



//---------------------------------------------//
// Iterate at each timer interupt
void ENCODER_PROCESSOR::iterateOnce(){                      // Moving average: calculating the speed of the motor for feedback control
    // new input
    MA_window[idx_MA_array] = pulses_;

    // idx_next: Actually, it is the oldest one.
    size_t idx_next = idx_MA_array + 1;
    if(idx_next > (size_MA_window - 1) ){idx_next = 0;}
    // Calculate the total number of pulses within the period
    delta_count = (MA_window[idx_MA_array] - MA_window[idx_next]);
    //
    idx_MA_array ++;
    if(idx_MA_array > (size_MA_window -1) ){idx_MA_array = 0;}
}
//---------------------------------------------//


// Get states
int ENCODER_PROCESSOR::getEncoderState(void) {
    return encoderState_now;
}

int ENCODER_PROCESSOR::getPulses(void) {
    return pulses_;
}

int ENCODER_PROCESSOR::getRevolutions(void) {
    return revolutions_;
}

// Get results
// Rotational speed
double ENCODER_PROCESSOR::getAngularSpeed(){
    return delta_count*count_2_rad_s;
}

double ENCODER_PROCESSOR::getAngularSpeed_deg_s(){
    return delta_count*count_2_deg_s;
}
// Angle
double ENCODER_PROCESSOR::getAngle(bool is_ranged){ // rad, if is_ranged, return 0~2*PI
    //
    int pulse_temp = this->pulses_;
    //
    if (is_ranged){
        revolutions_ = pulse_temp/pulsesPerRevolution;
        return (pulse_temp % pulsesPerRevolution)*count_2_rad;
    }else{
        return (pulse_temp*count_2_rad);
    }
}
double ENCODER_PROCESSOR::getAngle_deg(bool is_ranged){ // deg, if is_ranged, return 0~360
    //
    int pulse_temp = this->pulses_;
    //
    if (is_ranged){
        revolutions_ = pulse_temp/pulsesPerRevolution;
        return (pulse_temp % pulsesPerRevolution)*count_2_deg;
    }else{
        return (pulse_temp*count_2_deg);
    }
}
