/*===== NODE 2 =====*/

#include "mbed.h"
#include "main.h"
// SX1272 Lib
#include "sx1272-hal.h"
#include "debug.h"
// DHT11 Lib (air temperature and humidity)
#include "DHT11.h"
// SHT10 (soil temperature and humidity)
#include "SHTx/sht15.hpp"
// Shield capteur
#include "X_NUCLEO_IKS01A2/XNucleoIKS01A2.h"
// Sleep mode library
#include "WakeUp.h"

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1

/* DELAY between two transmission (in seconds) */
#define TRANSMISSION_DELAY                          1800

#define RF_FREQUENCY                                868000000 // Hz
#define TX_OUTPUT_POWER                             14        // 14 dBm

#define LORA_BANDWIDTH                              2         // [0: 125 kHz,
//  1: 250 kHz,
//  2: 500 kHz,
//  3: Reserved]

#define LORA_SPREADING_FACTOR                       7         // [SF7..SF12]
#define LORA_CODINGRATE                             1         // [1: 4/5,
//  2: 4/6,
//  3: 4/7,
//  4: 4/8]

#define LORA_PREAMBLE_LENGTH                        8
#define LORA_SYMBOL_TIMEOUT                         5         // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON                  false
#define LORA_FHSS_ENABLED                           false
#define LORA_NB_SYMB_HOP                            4
#define LORA_IQ_INVERSION_ON                        false
#define LORA_CRC_ENABLED                            true

#define BUFFER_SIZE                                 11         // Define the payload size here

#define KEY                                         0xA2



/* ===== PRECISION ===== */
#define PAS_temp                                    0.5
#define PAS_hum_a                                   5

/* ===== Période de mesure ===== */
#define T_mesure                                    3 /* 600 s */


DigitalOut led(LED1);
Serial pc(USBTX, USBRX);

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1272MB2xAS Radio( NULL );

uint8_t msg[BUFFER_SIZE];

uint16_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];

/* ===================== IKS01A2 ======================= */
/* Includes */
#include "XNucleoIKS01A2.h"
 
/* Instantiate the expansion board */
static XNucleoIKS01A2 *mems_expansion_board = XNucleoIKS01A2::instance(D14, D15, D4, D5);
 
/* Retrieve the composing elements of the expansion board */
static HTS221Sensor *hum_temp = mems_expansion_board->ht_sensor;
static LPS22HBSensor *press_temp = mems_expansion_board->pt_sensor;

 
/* ===================== Convertion ======================*/

// Air temperature and humidity sensor
DHT11 airSensor(D6);
int DHT11_state;

// Soil temperature and humidity sensor
SHTx::SHT15 soilSensor(D9, D8);

int nb_mesure = 0;


LowPowerTimeout MyTimeout;
bool expired = false;

void callback(void) {
    expired = true;
}


int main()
{
    //programAlarmA(20);
    WakeUp::calibrate();
    MyTimeout.attach(callback, T_mesure);
    
    while(1)
    {
        uint8_t id;
        float temp, hum_a, temp2, hum_s = 0xAA;
        char buffer1[32], buffer2[32];
        
        pc.printf( "\n\n\r >==== iGreenhouse Application - Transmitter ====< \n\n\r" );
        
        // IKS01A2
        /* Enable all sensors */
        hum_temp->enable();
        press_temp->enable();  
      
        hum_temp->read_id(&id);
        //pc.printf("\n\rHTS221  humidity & temperature    = 0x%X\n\r", id);
        press_temp->read_id(&id);
        //pc.printf("\n\rLPS22HB  pressure & temperature   = 0x%X\n\r", id);
    
        // Initialize Radio driver
        RadioEvents.TxDone = OnTxDone;
        RadioEvents.TxTimeout = OnTxTimeout;
        Radio.Init( &RadioEvents );
    
        // verify the connection with the board
        while( Radio.Read( REG_VERSION ) == 0x00  ) {
            debug( "\n\rRadio could not be detected!\n\r", NULL );
            wait( 1 );
        }
    
        debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1272MB2XAS ) ) , "\n\r > Board Type: SX1272MB2xAS < \n\r" );
    
        Radio.SetChannel( RF_FREQUENCY );
    
    
        debug_if( LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
        debug_if( !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");
    
        Radio.SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                           LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                           LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                           LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                           LORA_IQ_INVERSION_ON, 2000000 );
                           
        // Soil sensor configuration
        /*soilSensor.setOTPReload(false);
        soilSensor.setResolution(true);
        soilSensor.setScale(false);*/
    
        debug_if( DEBUG_MESSAGE, "Starting sending loop\r\n" );
    
        led = 0;

        // Retrieving sensors data
        DHT11_state = airSensor.readData();//DHT11
        soilSensor.update();//SHT15
        
        /* ============ IKS01A2 ==============*/
        pc.printf("\n\r === OK === \r\n");
        
        press_temp->get_temperature(&temp);
        hum_temp->get_humidity(&hum_a);
        hum_temp->get_temperature(&temp2);
        
        pc.printf("\n\r[temp] %7s C\r\n", print_double(buffer1, temp));
        pc.printf("\n\r[temp2] %7s C\r\n", print_double(buffer1, temp2));
        pc.printf("\n\r[hum] %s%%\r\n", print_double(buffer2, hum_a));
        
        temp*=5;
        hum_a*=2;
        
        pc.printf("\n\r[temp] %7s C\r\n", print_double(buffer1, temp));
        pc.printf("\n\r[hum] %s%%\r\n", print_double(buffer2, hum_a));
        
        
        msg[0] = 0x00;                      /* Identificateur Serres connectées */
        msg[1] = KEY;                       /* Identificateur Serres */
        msg[3*nb_mesure-1] = temp;          /* Température */
        msg[3*nb_mesure] = hum_a;           /* Humidité air */
        msg[3*nb_mesure+1] = hum_s;         /* Humidité sol */
        
        if(nb_mesure == 3)
        { 
            // Sending a new packet
            debug("\r\n========\r\nSending a new Packet\r\n========\r\n");
            for(int i = 0; i < BufferSize; i++) {
                pc.printf("%x", msg[i]);    
            }
            debug_if( DEBUG_MESSAGE, "\n" );
            memcpy( Buffer, msg, BufferSize );
            wait_ms(10);
            Radio.Send(Buffer, BufferSize);
            nb_mesure = 0;
        }
        
        temp/=5;
        hum_a/=2;

        
        // Switch the led state
        led = 1-led;

        nb_mesure++;
//BrunoS        while(!expired) sleep();
        expired=0;
    }
    
    pc.printf("Starting Standby mode...\r\n");
    HAL_PWR_EnterSTANDBYMode();
}

// temperature: -30 < x < 70
// humidity: 0 < x < 100
uint8_t to_u8(float x, bool isTemp)
{
  float a = 0;
  float min = 0.0;
  float max = 100.0;
  if( isTemp) {
    min = -30.0;
    max = 70.0;  
    a = 30.0;  
  }
  // On passe le float entre 0 et 1.0
  if(x > min && x < max) {
    x = (x + a) / 100.0;
  } else if(x <= min) {
    x = 0.0;
  } else {
    x = 1.0;
  }
  return rint(x * 255);
}

void OnTxDone( void )
{
    Radio.Sleep( );
    debug_if( DEBUG_MESSAGE, "> OnTxDone\n\r" );
}

void OnTxTimeout( void )
{
    Radio.Sleep( );
    debug_if( DEBUG_MESSAGE, "> OnTxTimeout\n\r" );
}









/* ================== IKS01A2 =====================*/

/* Helper function for printing floats & doubles */
static char *print_double(char* str, double v)
{
  int decimalDigits=2;
  int i = 1;
  int intPart, fractPart;
  int len;
  char *ptr;
 
  /* prepare decimal digits multiplicator */
  for (;decimalDigits!=0; i*=10, decimalDigits--);
 
  /* calculate integer & fractinal parts */
  intPart = (int)v;
  fractPart = (int)((v-(double)(int)v)*i);
 
  /* fill in integer part */
  sprintf(str, "%i.", intPart);
 
  /* prepare fill in of fractional part */
  len = strlen(str);
  ptr = &str[len];
 
  /* fill in leading fractional zeros */
  for (i/=10;i>1; i/=10, ptr++) {
    if (fractPart >= i) {
      break;
    }
    *ptr = '0';
  }
 
  /* fill in (rest of) fractional part */
  sprintf(ptr, "%i", fractPart);
 
  return str;
}