/// Calibration procedures for determining position sensor offset, 
/// phase ordering, and position sensor linearization
/// 

#include "calibration.h"
#include "foc.h"
#include "PreferenceWriter.h"
#include "user_config.h"
#include "math.h"
#include "math_ops.h"

void measure_rl(int n, GPIOStruct *gpio, ControllerStruct *controller, PreferenceWriter *prefs){
    
    
    }

void order_phases(PositionSensor *ps, GPIOStruct *gpio, ControllerStruct *controller, PreferenceWriter *prefs){   
    
    ///Checks phase order, to ensure that positive Q current produces
    ///torque in the positive direction wrt the position sensor.
    printf("\n\r Checking phase ordering\n\r");
    float theta_ref = 0;
    float theta_actual = 0;
    float v_d = 2.0;                                                             //Put all volts on the D-Axis
    float v_q = 0.0;
    float v_u, v_v, v_w = 0;
    float dtc_u, dtc_v, dtc_w = .5;
    int sample_counter = 0;
    
    ///Set voltage angle to zero, wait for rotor position to settle
    abc(theta_ref, v_d, v_q, &v_u, &v_v, &v_w);                                 //inverse dq0 transform on voltages
    svm(V_BUS, v_u, v_v, v_w, &dtc_u, &dtc_v, &dtc_w);                            //space vector modulation
    TIM1->CCR3 = (PWM_ARR)*(0.5f);                                        // Set duty cycles
    TIM1->CCR2 = (PWM_ARR)*(0.5f);
    TIM1->CCR1 = (PWM_ARR)*(0.5f);
    wait_us(100);
    for(int i = 0; i<20000; i++){
        wait_us(50);
        TIM1->CCR3 = (PWM_ARR)*(1.0f-dtc_u);                                        // Set duty cycles
        TIM1->CCR2 = (PWM_ARR)*(1.0f-dtc_v);
        TIM1->CCR1 = (PWM_ARR)*(1.0f-dtc_w);
        }
        
    
    //ps->ZeroPosition();
    ps->Sample(); 
    wait_us(1000);
    //float theta_start = ps->GetMechPosition();                                  //get initial rotor position
    float theta_start;
    
    /// Rotate voltage angle
    while(theta_ref < 4.0f*PI){                                                    //rotate for 2 electrical cycles
        abc(theta_ref, v_d, v_q, &v_u, &v_v, &v_w);                             //inverse dq0 transform on voltages
        svm(V_BUS, v_u, v_v, v_w, &dtc_u, &dtc_v, &dtc_w);                        //space vector modulation
        wait_us(100);
        TIM1->CCR3 = (PWM_ARR)*(1.0f-dtc_u);                                        //Set duty cycles
        TIM1->CCR2 = (PWM_ARR)*(1.0f-dtc_v);
        TIM1->CCR1 = (PWM_ARR)*(1.0f-dtc_w);
       ps->Sample();                                                            //sample position sensor
       theta_actual = ps->GetMechPosition();
       if(theta_ref==0){theta_start = theta_actual;}
       if(sample_counter > 200){
           sample_counter = 0 ;
        printf("%.4f   %.4f\n\r", theta_ref/(NPP), theta_actual);
        }
        sample_counter++;
       theta_ref += 0.001f;
        }
    float theta_end = ps->GetMechPosition();
    int direction = (theta_end - theta_start)>0;
    float ratio = abs(4.0f*PI/(theta_end-theta_start));
    int pole_pairs = (int) roundf(ratio);
        
    printf("Theta Start:   %f    Theta End:  %f\n\r", theta_start, theta_end);
    printf("Direction:  %d\n\r", direction);
    if(direction){printf("Phasing correct\n\r");}
    else if(!direction){printf("Phasing incorrect.  Swapping phases V and W\n\r");}
    PHASE_ORDER = direction;
    }
    
    
void calibrate(PositionSensor *ps, GPIOStruct *gpio, ControllerStruct *controller, PreferenceWriter *prefs){
    /// Measures the electrical angle offset of the position sensor
    /// and (in the future) corrects nonlinearity due to position sensor eccentricity
    printf("Starting calibration procedure\n\r");
    
    const int n = 128*NPP;                                                      // number of positions to be sampled per mechanical rotation.  Multiple of NPP for filtering reasons (see later)
    const int n2 = 50;                                                          // increments between saved samples (for smoothing motion)
    float delta = 2*PI*NPP/(n*n2);                                              // change in angle between samples
    float error_f[n] = {0};                                                     // error vector rotating forwards
    float error_b[n] = {0};                                                     // error vector rotating backwards
    const int  n_lut = 128;
    int lut[n_lut]= {0};                                                        // clear any old lookup table before starting.
    ps->WriteLUT(lut); 
    int raw_f[n] = {0};
    int raw_b[n] = {0};
    float theta_ref = 0;
    float theta_actual = 0;
    float v_d = 2.0f;                                                             // Put volts on the D-Axis
    float v_q = 0.0;
    float v_u, v_v, v_w = 0;
    float dtc_u, dtc_v, dtc_w = .5;
    
        
    ///Set voltage angle to zero, wait for rotor position to settle
    abc(theta_ref, v_d, v_q, &v_u, &v_v, &v_w);                                 // inverse dq0 transform on voltages
    svm(V_BUS, v_u, v_v, v_w, &dtc_u, &dtc_v, &dtc_w);                            // space vector modulation
    for(int i = 0; i<40000; i++){
        TIM1->CCR3 = (PWM_ARR)*(1.0f-dtc_u);                                        // Set duty cycles
        if(PHASE_ORDER){                                   
            TIM1->CCR2 = (PWM_ARR)*(1.0f-dtc_v);
            TIM1->CCR1 = (PWM_ARR)*(1.0f-dtc_w);
            }
        else{
            TIM1->CCR1 = (PWM_ARR)*(1.0f-dtc_v);
            TIM1->CCR2 = (PWM_ARR)*(1.0f-dtc_w);
            }
        wait_us(100);
        }
    ps->Sample();   
    printf(" Current Angle : Rotor Angle : Raw Encoder \n\r\n\r");
    for(int i = 0; i<n; i++){                                                   // rotate forwards
       for(int j = 0; j<n2; j++){   
        theta_ref += delta;
       abc(theta_ref, v_d, v_q, &v_u, &v_v, &v_w);                              // inverse dq0 transform on voltages
       svm(V_BUS, v_u, v_v, v_w, &dtc_u, &dtc_v, &dtc_w);                         // space vector modulation
        TIM1->CCR3 = (PWM_ARR)*(1.0f-dtc_u);
        if(PHASE_ORDER){                                                        // Check phase ordering
            TIM1->CCR2 = (PWM_ARR)*(1.0f-dtc_v);                                    // Set duty cycles
            TIM1->CCR1 = (PWM_ARR)*(1.0f-dtc_w);
            }
        else{
            TIM1->CCR1 = (PWM_ARR)*(1.0f-dtc_v);
            TIM1->CCR2 = (PWM_ARR)*(1.0f-dtc_w);
            }
            wait_us(100);
            ps->Sample();
        }
       ps->Sample();
       theta_actual = ps->GetMechPosition();
       error_f[i] = theta_ref/NPP - theta_actual;
       raw_f[i] = ps->GetRawPosition();
        printf("%.4f   %.4f    %d\n\r", theta_ref/(NPP), theta_actual, raw_f[i]);
       //theta_ref += delta;
        }
    
    for(int i = 0; i<n; i++){                                                   // rotate backwards
       for(int j = 0; j<n2; j++){
       theta_ref -= delta;
       abc(theta_ref, v_d, v_q, &v_u, &v_v, &v_w);                              // inverse dq0 transform on voltages
       svm(V_BUS, v_u, v_v, v_w, &dtc_u, &dtc_v, &dtc_w);                         // space vector modulation
        TIM1->CCR3 = (PWM_ARR)*(1.0f-dtc_u);
        if(PHASE_ORDER){
            TIM1->CCR2 = (PWM_ARR)*(1.0f-dtc_v);
            TIM1->CCR1 = (PWM_ARR)*(1.0f-dtc_w);
            }
        else{
            TIM1->CCR1 = (PWM_ARR)*(1.0f-dtc_v);
            TIM1->CCR2 = (PWM_ARR)*(1.0f-dtc_w);
            }
            wait_us(100);
            ps->Sample();
        }
       ps->Sample();                                                            // sample position sensor
       theta_actual = ps->GetMechPosition();                                    // get mechanical position
       error_b[i] = theta_ref/NPP - theta_actual;
       raw_b[i] = ps->GetRawPosition();
       printf("%.4f   %.4f    %d\n\r", theta_ref/(NPP), theta_actual, raw_b[i]);
       //theta_ref -= delta;
        }    
        
        float offset = 0;                                  
        for(int i = 0; i<n; i++){
            offset += (error_f[i] + error_b[n-1-i])/(2.0f*n);                   // calclate average position sensor offset
            }
        offset = fmod(offset*NPP, 2*PI);                                        // convert mechanical angle to electrical angle
        
            
        ps->SetElecOffset(offset);                                              // Set position sensor offset
        __float_reg[0] = offset;
        E_OFFSET = offset;
        
        /// Perform filtering to linearize position sensor eccentricity
        /// FIR n-sample average, where n = number of samples in one electrical cycle
        /// This filter has zero gain at electrical frequency and all integer multiples
        /// So cogging effects should be completely filtered out.
        
        float error[n] = {0};
        const int window = 128;
        float error_filt[n] = {0};
        float mean = 0;
        for (int i = 0; i<n; i++){                                              //Average the forward and back directions
            error[i] = 0.5f*(error_f[i] + error_b[n-i-1]);
            }
        for (int i = 0; i<n; i++){
            for(int j = 0; j<window; j++){
                int ind = -window/2 + j + i;                                    // Indexes from -window/2 to + window/2
                if(ind<0){
                    ind += n;}                                                  // Moving average wraps around
                else if(ind > n-1) {
                    ind -= n;}
                error_filt[i] += error[ind]/(float)window;
                }

            //printf("%.4f   %4f    %.4f   %.4f\n\r", error[i], error_filt[i], error_f[i], error_b[i]);
            mean += error_filt[i]/n;
            }
        int raw_offset = (raw_f[0] + raw_b[n-1])/2;                             //Insensitive to errors in this direction, so 2 points is plenty
        
        
        printf("\n\r Encoder non-linearity compensation table\n\r");
        printf(" Sample Number : Lookup Index : Lookup Value\n\r\n\r");
        for (int i = 0; i<n_lut; i++){                                          // build lookup table
            int ind = (raw_offset>>7) + i;
            if(ind > (n_lut-1)){ 
                ind -= n_lut;
                }
            lut[ind] = (int) ((error_filt[i*NPP] - mean)*(float)(ps->GetCPR())/(2.0f*PI));
            printf("%d   %d   %d\n\r", i, ind, lut[ind]);
            wait(.001);
            }
            
        ps->WriteLUT(lut);                                                      // write lookup table to position sensor object
        memcpy(&ENCODER_LUT, lut, sizeof(lut));                                 // copy the lookup table to the flash array
        printf("\n\rEncoder Electrical Offset (rad) %f\n\r",  offset);
        
        if (!prefs->ready()) prefs->open();
        prefs->flush();                                                         // write offset and lookup table to flash
        prefs->close();


    }