// ROUTER
#include "ConfigFile.h"
#include "mbed.h"

#define MMA8452_CTRL_REG_1 0x2A
#define MMA8452_ADDRESS 0x3A
#define MMA8452_OUT_Z_MSB 0x05 // [7:0] are 8 MSBs of 12-bit sample
#define MMA8552_OUT_Z_LSB 0x06 // [7:4] are 4 LSBs of 12-bit sample
#define MMA8452_WHO_AM_I 0x0D

DigitalOut reset(p8);
DigitalIn btn(p15);
I2C i2c(p28, p27); // SDA, SCL
Serial xbee(p13, p14);  // tx, rx
Serial pc(USBTX, USBRX); // tx, rx
Ticker ticker;

int16_t (*functionPointers[2])();
uint16_t panId = 0;
char panIdChar[5];

void printBuffer(uint8_t bufferSize, uint8_t* buffer) {
    for(int k = 0; k < bufferSize; k++) {
        pc.printf("%X-", buffer[k]);
    }
    printf("\n");
}

void readConfigFile() {
    LocalFileSystem local("local");
    ConfigFile cfg;
    char *panIdStr = "panID";
    
    if (!cfg.read("/local/input.cfg")) { 
        error("Erreur dans la lecture du fichier de configuration.\n");
    } else {
        cfg.getValue(panIdStr, &panIdChar[0], sizeof(panIdChar));
        panId = (uint16_t)strtol(panIdChar, NULL, 10);
        printf("The PAN ID is %i\n", panId);
    }
}

int16_t readXYZAcceleration() {
    char addr[1] = { MMA8452_OUT_Z_MSB };
    char data[1] = { 0 };
    char z[2] = { 0 };
    char buff[6] = { 0 };
    int16_t temp; 
    
    data[0] = MMA8452_WHO_AM_I;
    i2c.write(MMA8452_ADDRESS, data, 1, true);
    i2c.read(MMA8452_ADDRESS, buff, 1);
    
    if (buff[0] != 0x2A){
        printf("Erreur de communication...");
        return 1;
    }

    // Set actif mode on
    buff[0] = 0;
    data[0] = MMA8452_CTRL_REG_1;
    i2c.write(MMA8452_ADDRESS, data, 1, true);
    i2c.read(MMA8452_ADDRESS, &buff[0], 1);
    buff[0] |= 0x05; 
    i2c.write(MMA8452_ADDRESS, buff, 1);
    
    i2c.write(MMA8452_ADDRESS, addr, 1, true);
    i2c.read(MMA8452_ADDRESS, &z[0], 2); // OUT_Z_MSB, and OUT_Z_LSB

    // MSB + LSB
    ((char*)&temp)[0] = z[1];
    ((char*)&temp)[1] = z[0];
    return (temp >> 4);
}

int16_t readDryContact() {
    return btn;
}

uint8_t calculateChecksum(uint8_t frameSize, uint8_t* buffer) {
    uint32_t checksum = 0;
    uint8_t frameTypeIndex = 3;
    
    for (int i = frameTypeIndex; i < frameSize - 1; i++) {
        checksum += buffer[i];
    }

    return (uint8_t)(0xFF - (checksum & 0xFF));      
}

void sendFrame(uint8_t bufferSize, uint8_t* buffer) {
    for (int i = 0; i < bufferSize;) {
        if (xbee.writeable()) {
            xbee.putc(buffer[i]);
            i++;
        }
    }
}

void setFrame(bool dryContact, int16_t accelerations) {
    const uint8_t bufferSize = 22;
    uint8_t buffer[bufferSize] = { 0 };
    
    buffer[0] = 0x7E; // Start Delimiter
    buffer[1] = 0x00; // Length
    buffer[2] = 0x12; // Length
    buffer[3] = 0x10; // Frame Type
    buffer[4] = 0x01; // Frame ID
    buffer[5] = 0x00; // 64 bit adress
    buffer[6] = 0x00; // 64 bit adress
    buffer[7] = 0x00; // 64 bit adress
    buffer[8] = 0x00; // 64 bit adress
    buffer[9] = 0x00; // 64 bit adress
    buffer[10] = 0x00; // 64 bit adress
    buffer[11] = 0x00; // 64 bit adress
    buffer[12] = 0x00; // 64 bit adress
    buffer[13] = 0xFF; // 16 bit adress
    buffer[14] = 0xFE; // 16 bit adress
    buffer[15] = 0x00; // Broadcast radius
    buffer[16] = 0x00; // Options
    buffer[17] = 0x02; // RF Data...
    buffer[18] = 0x02; // RF Data...
    buffer[19] = 0x02; // RF Data...
    buffer[20] = 0x02; // RF Data...
    buffer[21] = calculateChecksum(bufferSize, buffer); // Checksum
    
    printBuffer(bufferSize, buffer);
    sendFrame(bufferSize, buffer);
}

void sendDataToCoordinator() {
    bool dryContact = functionPointers[0] ();
    int16_t XYZAccelerations = functionPointers[1]();
        
    printf("Contact sec: %d, Acceleration: %i\n", dryContact, XYZAccelerations); 
    setFrame(dryContact, XYZAccelerations);
}

int main() {
    printf("Starting a router...\n");    
    
    reset = 0;
    wait_ms(1);
    reset = 1;
    wait_ms(1);
    
    readConfigFile();
    
    xbee.baud(9600);                    // Set baud rate
    xbee.printf("ATID %i\r", panId);    // Set the 64-bit PAN ID
    xbee.printf("ATWR \r");
    xbee.printf("ATCN \r");
    
    functionPointers[0] = readDryContact;       // a. Un contact sec
    functionPointers[1] = readXYZAcceleration;    // b. Un accéléromètre 3 axes
    
    ticker.attach(&sendDataToCoordinator, 1.0);
    while (1) {}
}
