/* mbed Minimu_9 Library version 0beta1
 * Copyright (c) 2012 bengo
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef minimu_h
#define minimu9_h

#include "mbed.h"
#include <LSM303.h>
#include <L3G4200D.h>
#include <vector>

/** Minimu-9 accelerometer, magnetometer, gyroscope sensor controller library
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include <minimu9.h>
 * #include <vector>
 * 
 * Serial pc(USBTX, USBRX); // tx, rx
 * 
 * int main() {
 *     
 *    minimu9 minimu( p28, p27 );
 *    
 *    int status = minimu.getStatus();
 *    if( status == 0 ) {
 *       for( int i=0; i<100000; i++ ) {
 *          std::vector<int> data = minimu.read();
 *          if( status == 0 ) {    
 *             pc.printf("\x1b[2J\x1b[f\x1b[33mLSM303 minimu read status: \x1b[32m%d\n\r", minimu.getStatus() );
 *          }
 *          else {
 *             pc.printf("\x1b[2J\x1b[f\x1b[33mLSM303 minimu read status: \x1b[31m%d\n\r", minimu.getStatus() );
 *          }
 *          
 *          pc.printf("\x1b[33mLSM303   acc: \x1b[37m%8d %8d %8d\n\r", (short)data[0], (short)data[1], (short)data[2] );
 *          pc.printf("\x1b[33mLSM303   mag: \x1b[37m%8d %8d %8d\n\r", (short)data[3], (short)data[4], (short)data[5] );
 *          pc.printf("\x1b[33mL3G4200D gyr: \x1b[37m%8d %8d %8d\n\r", (short)data[6], (short)data[7], (short)data[8] );
 *          wait( 1 );
 *       }
 *    }
 *    return( 0 );
 * }
 * @endcode
*/

class minimu9 {

public:
  
   /**
    * Create a minimu9 object connected to the specified I2C pins
    * @param sda I2C SDA pin
    * @param scl I2C SCL pin
    */
   minimu9( PinName sda, PinName scl );
   
   /**
    * Return status code of prevoius function call
    */
   inline int getStatus( void ) { return( _status ); }

   /**
    * Read all three sensors vectors (accelerometer, magnetometer, gyroscope)
    */
   std::vector<short> read( void );
   
    /**
     * Read accelerometer vector
     */
   inline std::vector<short> accRead( void ) { return( _lsm.accRead() ); }
   
   /**
    * Read acceleration (in g units)
    */
   inline std::vector<float> acceleration( void ) { return( _lsm.acceleration() ); } 
   
   /**
    * Read magnetometer vector
    */
   inline std::vector<short> magRead( void ) { return( _lsm.magRead() ); }
   
   /**
    * Read magnetic filed (in gauss)
    */
   inline std::vector<float> magneticField( void ) { return( _lsm.magneticField() ); }
   
   /**
    * Read gyroscope vector
    */
   inline std::vector<short> gyrRead( void ) { return( _l3g.read() ); }
   
   /**
    * Read angular velogity (in degrees per second)
    */
   inline std::vector<float> angularVelocity( void ) { return( _l3g.angularVelocity() ); }
   
    /**
    * Read temperature (in celsius)
    */
   inline int temperature( void ) { return( _l3g.temperature() ); }
   
   /**
    * Return LSM303 associated object
    */ 
   inline LSM303 getLSM303( void ) { return( _lsm ); }
   
   /**
    * Return L3G4200D associated object
    */ 
   inline L3G4200D getL3G4200D( void ) { return( _l3g ); }
   
private:

   int _status;   
   I2C _i2c;
   LSM303 _lsm;
   L3G4200D _l3g;
  
}; 

#endif // minimu9_h