/* mbed LSM303 Library version 0beta1
 * Copyright (c) 2012 bengo
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#include <LSM303.h>
#include <cmath>

// LSM303 I2C addresses
const int LSM303::ACC_ADDRESS = 0x30;
const int LSM303::MAG_ADDRESS = 0x3c;
// LSM303 register addresses
const int LSM303::ACC_CTRL_REG1 = 0x20;
const int LSM303::ACC_CTRL_REG2 = 0x21;
const int LSM303::ACC_CTRL_REC3 = 0x22;
const int LSM303::ACC_CTRL_REG4 = 0x23;
const int LSM303::ACC_CTRL_REG5 = 0x24;
const int LSM303::ACC_HP_FILTER_RESET = 0x25;
const int LSM303::ACC_REFERENCE = 0x26;
const int LSM303::ACC_STATUS_REG = 0x27;
const int LSM303::ACC_OUT_X_L = 0x28;
const int LSM303::ACC_OUT_X_H = 0x29;
const int LSM303::ACC_OUT_Y_L = 0x2a;
const int LSM303::ACC_OUT_Y_H = 0x2b;
const int LSM303::ACC_OUT_Z_L = 0x2c;
const int LSM303::ACC_OUT_Z_H = 0x2d;
const int LSM303::ACC_INT1_CFG = 0x30;
const int LSM303::ACC_INT1_SOURCE = 0x31;
const int LSM303::ACC_INT1_THS = 0x32;
const int LSM303::ACC_INT1_DURATION = 0x33;
const int LSM303::ACC_INT2_CFG = 0x34;
const int LSM303::ACC_INT2_SOURCE = 0x35;
const int LSM303::ACC_INT2_THS = 0x36;
const int LSM303::ACC_INT2_DURATION = 0x37;
const int LSM303::MAG_CRA_REG = 0x00;
const int LSM303::MAG_CRB_REG = 0x01;
const int LSM303::MAG_MR_REG = 0x02;
const int LSM303::MAG_OUT_X_H = 0x03;
const int LSM303::MAG_OUT_X_L = 0x04;
const int LSM303::MAG_OUT_Y_H = 0x07;
const int LSM303::MAG_OUT_Y_L = 0x08;
const int LSM303::MAG_OUT_Z_H = 0x05;
const int LSM303::MAG_OUT_Z_L = 0x6;
const int LSM303::MAG_SR_REG = 0x9;
const int LSM303::MAG_IRA_REG = 0xa;
const int LSM303::MAG_IRB_REG = 0xb;
const int LSM303::MAG_IRC_REG = 0xc;
const int LSM303::MAG_WHO_AM_I = 0xf;
//

// -------------------------------------------
LSM303::LSM303( PinName sda, PinName scl ) : _i2c( sda, scl ) {
   // Get SA0 pin status
   _bytes[0] = ACC_CTRL_REG1;
   _i2c.write( ACC_ADDRESS, _bytes, 1 );
   int sa0low = _i2c.read( ACC_ADDRESS+1, _bytes, 1 );
   _bytes[0] = ACC_CTRL_REG1;
   _i2c.write( ACC_ADDRESS+2, _bytes, 1 );
   int sa0hig = _i2c.read( ACC_ADDRESS+2+1, _bytes, 1 );
   if( sa0low == 0 && sa0hig != 0 ) {
      _SA0Pad = 0x0;
   }
   else if( sa0low != 0 && sa0hig == 0 ) {
      _SA0Pad = 0x2;
   }
   else {
      _status = 1;
      return;
   }
   // Check that you're talking with an LM303DLM device
   _bytes[0] = MAG_WHO_AM_I;
   _i2c.write( MAG_ADDRESS, _bytes, 1 );
   _status = _i2c.read( MAG_ADDRESS+1, _bytes, 1 );   
   if( _bytes[0] == 0x3c ) {
      _status = 0;
   }
   else {
      _status = 1;
      return;
   }
   // Enable normal mode... 
   // ... On accelerometer
   this->accRegisterWrite( ACC_CTRL_REG1, 0x27 );
   if( _status != 0 ) {
      return;  
   }
   // ... And on magnetometer
   this->magRegisterWrite( MAG_MR_REG, 0x00 );
}

// -------------------------------------------
int LSM303::accRegisterRead( int reg ) {
  _bytes[0] = reg & 0xff;
  _status = _i2c.write( ACC_ADDRESS + _SA0Pad, _bytes, 1 );
  if( _status ==  0 ) {
    _status = _i2c.read(  ACC_ADDRESS + _SA0Pad + 1, _bytes, 1 );
    return( _bytes[0] );
  }
  return( 0 );
}

// -------------------------------------------  
void LSM303::accRegisterWrite( int reg, char data ) {
  _bytes[0] = reg & 0xff;
  _bytes[1] = data & 0xff;
  _status = _i2c.write( ACC_ADDRESS + _SA0Pad, _bytes, 2 );
}

// -------------------------------------------  
int LSM303::magRegisterRead( int reg ) {
  _bytes[0] = reg & 0xff;
  _status = _i2c.write( MAG_ADDRESS, _bytes, 1 );
  if( _status ==  0 ) {
    _status = _i2c.read(  MAG_ADDRESS + 1, _bytes, 1 );
    return( _bytes[0] );
  }
  return( 0 );
}

// -------------------------------------------  
void LSM303::magRegisterWrite( int reg, char data ) {
  _bytes[0] = reg & 0xff;
  _bytes[1] = data & 0xff;
  _status = _i2c.write( MAG_ADDRESS, _bytes, 2 );
}


// -------------------------------------------
std::vector<short> LSM303::accRead( void ) {
   std::vector<short> acc( 3, 0 );
   _bytes[0] = ACC_OUT_X_L | (1<<7);
   _status = _i2c.write( ACC_ADDRESS + _SA0Pad, _bytes, 1 );
   if( _status == 0 ) {
      _status = _i2c.read( ACC_ADDRESS + _SA0Pad + 1, _bytes, 6 );
      if( _status == 0 ) {
         for( int i=0; i<3; i++ ) {
            acc[i] = ( short( _bytes[2*i] ) | short(_bytes[2*i+1]) << 8 );
         }
      }
   } 
   return( acc );
}

// -------------------------------------------
std::vector<float> LSM303::acceleration( void ) {

   const float cal[3][2] = { {  16291.5, -16245.4 }, {  16819.0, -16253.0 }, {  16994.8, -15525.6 } };
   
   std::vector<float> acc( 3, 0 );
   int fs = ( this->accRegisterRead( ACC_CTRL_REG4 ) >> 4 ) & 0x3;
   std::vector<short> a = this->accRead();
   if( _status == 0 ) {
      for( int i=0; i<3; i++ ) {
         acc[i] = acc[i] * ( (cal[i][0] - cal[i][1]) / 32768. ) + (cal[i][0]+cal[i][1])/2.;
         acc[i] = float( a[i] ) * pow(2.,(fs+1)) / 32768.;
      }
   }
   return( acc );
}  

// -------------------------------------------
std::vector<short> LSM303::magRead( void ) {
   std::vector<short> mag( 3, 0 );
   _bytes[0] = MAG_OUT_X_H;
   _status = _i2c.write( MAG_ADDRESS, _bytes, 1 );
   if( _status == 0 ) {
      _status = _i2c.read( MAG_ADDRESS + 1, _bytes, 6 );
      if( _status == 0 ) {
         mag[0] = short( _bytes[0] << 8 ) | short( _bytes[1] );
         mag[1] = short( _bytes[4] << 8 ) | short( _bytes[5] );
         mag[2] = short( _bytes[2] << 8 ) | short( _bytes[3] );
      }
   } 
   return( mag );
}
 
// ------------------------------------------- 
std::vector<float> LSM303::magneticField( void ) {

   float gainxy[] = { 1100., 855., 670., 450., 400., 330., 230. };
   float gainz[]  = {  980., 760., 600., 400., 355., 295., 205. };
   
   std::vector<float> mag( 3, 0 );
   int gn = ( this->magRegisterRead( MAG_CRB_REG ) >> 5 ) & 0x7;
   std::vector<short> m = this->magRead();
   if( _status == 0 ) {
      mag[0] = float( m[0] ) / gainxy[gn-1];
      mag[1] = float( m[1] ) / gainxy[gn-1];
      mag[2] = float( m[2] ) / gainz[gn-1];
   }
   return( mag );
}