/*==============================================================================
* Edit History
* 
* This section contains comments describing changes made to the module. Notice
* that changes are listed in reverse chronological order. Please use ISO format
* for dates.
* 
* when       who       what, where, why
* ---------- ---       -----------------------------------------------------------
* 2016-10-20 bh        - Add flags: pah_stream_e, pah_intshape_pulse_type_e, pah_powerdown_int_status_e.
*                      - Deprecate error codes: pah_no_interrupt, pah_err_comm_fail.
*                      - Initial revision.
==============================================================================*/

#ifndef __pah_driver_types_h__
#define __pah_driver_types_h__


#include "pah_ret.h"
#include "pah_platform_types.h"


/**
 * @brief Data stream modes
 *
 * @var pah_stream_dri
 * @brief When the sample number of device FIFO buffer is more than preset value, the device will raise a FIFO interrupt. Then the client calls pah_task() to collect sample data from device.
 *
 * @var pah_stream_polling
 * @brief The device would never raise FIFO interrupts. The client must call pah_task() periodically, and the sample number depends on time interval between pah_task() calls.
 * 
 */
typedef enum {

    pah_stream_default,
    pah_stream_dri = pah_stream_default,    // Data ready interrupt
    pah_stream_polling,                     // Polling

} pah_stream_e;


/**
 * @brief Determine the interrupt shape pulled by device
 *
 * On some platforms, with pah_intshape_pulse_type_edge the pulse width may be too short to be identified.
 * In this case pah_intshape_pulse_type_level would be useful.
 *
 * @var pah_intshape_pulse_type_edge
 * @brief The device pull interrupt up then down or down then up.
 *
 * @var pah_intshape_pulse_type_level
 * @brief The device pull up or down, then the client calls pah_task() to pull back.
 *
 */
typedef enum {

    pah_intshape_pulse_type_default,
    pah_intshape_pulse_type_edge = pah_intshape_pulse_type_default,   // Edge-sensitive
    pah_intshape_pulse_type_level,                                    // Level-sensitive

} pah_intshape_pulse_type_e;


/**
 * @brief Determine the interrupt pin status when device powerdown.
 *
 * Some platforms doesn't support to pull up/down input gpio pins.
 * In this case pah_powerdown_int_status_down would be useful to avoid leakage current.
 *
 * @var pah_powerdown_int_status_floating
 * @brief The device doesn't pull interrupt pin in powerdown.
 *
 * @var pah_powerdown_int_status_down
 * @brief The device pull down interrupt pin in powerdown.
 *
 */
typedef enum {

    pah_powerdown_int_status_default,
    pah_powerdown_int_status_floating = pah_powerdown_int_status_default,   // Floating
    pah_powerdown_int_status_down,                                          // Pull down

} pah_powerdown_int_status_e;


/**
 * @brief The moment to turn LED on when PPG is enabled
 *
 * In some applications, PPG device would be enabled automatically, it's unnatural if the end user is not using.
 * In this case pah_ppg_led_on_deferred would be useful.
 *
 * @var pah_ppg_led_on_instantly
 * @brief When PPG is enabled, turn LED on instantly.
 *
 * @var pah_ppg_led_on_deferred
 * @brief When PPG is enabled, LED keeps off until first touch detection.
 *
 */
typedef enum {

    pah_ppg_led_on_default,
    pah_ppg_led_on_instantly = pah_ppg_led_on_default,    // Turn on LED when enabled PPG
    pah_ppg_led_on_deferred,                              // Turn on LED when enabled PPG and detected first touch 

} pah_ppg_led_on_e;


typedef enum {

    pah_none,                   // Not used

    pah_stop_mode,              // Stop mode

    pah_touch_mode,             // Touch mode

    pah_ppg_mode,               // PPG mode
    pah_ppg_200hz_mode,         // PPG(200Hz) mode

    pah_ppg_touch_mode,         // PPG + Touch mode
    pah_ppg_200hz_touch_mode,   // PPG(200Hz) + Touch mode

} pah_mode;


typedef enum {

    pah_device_ppg,             // PPG device
    pah_device_touch,           // Touch device

} pah_device;


// legacy pah_ret
#define pah_success                 PAH_RET_SUCCESS             // Success
#define pah_pending                 PAH_RET_SUCCESS             // Do nothing. Usually happens when task() detected no interrupt in dri mode or fifo number is not enough.
#define pah_no_interrupt            PAH_RET_SUCCESS
#define pah_err_unknown             PAH_RET_FAILED              // Unknown error
#define pah_err_invalid_argument    PAH_RET_INVALID_ARGUMENT    // Invalid argument to function
#define pah_err_invalid_operation   PAH_RET_INVALID_OPERATION   // Invalid operation to function
#define pah_err_not_init            PAH_RET_INVALID_OPERATION   // Driver isn't initialized
#define pah_err_not_implement       PAH_RET_NOT_IMPLEMENTED     // Not implemented in this driver
#define pah_err_platform_fail       PAH_RET_PLAT_FAILED         // Platform function failed (For example, I2C, SPI, ...)
#define pah_err_comm_fail           PAH_RET_PLAT_FAILED         // Platform function failed (For example, I2C, SPI, ...)
#define pah_err_invalid_program     PAH_RET_FAILED              // Generally this indicates a bug in the driver
#define pah_err_verify_device_fail  PAH_RET_VERIFY_FAILED       // Verify device failed
#define pah_err_fifo_checksum_fail  PAH_RET_VERIFY_FAILED       // Failed to compare check sum with fifo data, usually happens to bad data communication
#define pah_err_fifo_overflow       PAH_RET_FIFO_OVERFLOW       // Fifo in device is overflow, usually happens when pah_task() was too late to be called
#define pah_err_fifo_underflow      PAH_RET_FAILED              // Fifo in device is underflow


typedef struct {

    uint8_t*    data;
    uint8_t     touch_flag;

} pah_report_fifo;

typedef void(*pah_report_fifo_handle)(void* user_data, pah_report_fifo* fifo);



#endif  // header guard
