/*==============================================================================
* Edit History
*
* This section contains comments describing changes made to the module. Notice
* that changes are listed in reverse chronological order. Please use ISO format
* for dates.
*
* when       who       what, where, why
* ---------- ---       -----------------------------------------------------------
* 2016-10-20 bh        - Initial revision.
==============================================================================*/

#include "pah_8011_internal.h"

// pah
#include "pah_comm.h"

// platform support
#include "pah_platform_functions.h"


//============================================================================
// PUBLIC FUNCTION DEFINITIONS
//============================================================================
pah_ret pah_8011_verify_product_id(void)
{
    pah_ret ret = pah_err_unknown;
    uint8_t data = 0;

    if (!pah_comm_write(0x7F, 0x00))    //bank0
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }
    if (!pah_comm_read(0x00, &data))
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }
    
    if (data != PAH_PRODUCT_ID)
    {
        debug_printf_1("pah_8011_verify_product_id() fail. id = 0x%X \n", data);
        ret = pah_err_verify_device_fail;
        goto FAIL;
    }
    
    debug_printf_1("pah_8011_verify_product_id() success. id = 0x%X \n", data);

    return pah_success;

FAIL:
    debug_printf_1("pah8011_verify_product_id() failed. ret = %d \n", ret);
    return ret;
}

pah_ret pah_8011_write_touch_package_setting(void)
{
    pah_ret ret = pah_err_unknown;
    device_package_e device_package;

    ret = pah_8011_read_device_package(&device_package);
    if (PAH_FAILED(ret))
        goto FAIL;

    switch (device_package)
    {
    case device_package_et:
        {
            if (!pah_comm_write(0x7F, 0x01))    //bank1
            {
                ret = pah_err_platform_fail;
                goto FAIL;
            }
            if (!pah_comm_write(0x05, 0x04))    // Cmd_TouchDetection_Th_Hi_15_8
            {
                ret = pah_err_platform_fail;
                goto FAIL;
            }
            if (!pah_comm_write(0x08, 0x04))    // Cmd_TouchDetection_Th_Lo_15_8
            {
                ret = pah_err_platform_fail;
                goto FAIL;
            }
            if (!pah_comm_write(0x14, 0x06))    // Cmd_adc_osr_t
            {
                ret = pah_err_platform_fail;
                goto FAIL;
            }
            if (!pah_comm_write(0x75, 0x00))   // T_ledbias_sel
            {
                ret = pah_err_platform_fail;
                goto FAIL;
            }
        }
        break;

    case device_package_es:
        //// No need to modify, because these values are default or defined in init settings.
        // switch to bank1
        //Cmd_TouchDetection_Th_Hi_15_8 = 0x07;
        //Cmd_TouchDetection_Th_Lo_15_8 = 0x06;
        //Cmd_adc_osr_t                 = 0x03;
        //T_ledbias_sel                 = 0x03;
        break;

    default:
        ret = pah_err_not_implement;
        goto FAIL;
    }

    return pah_success;

FAIL:
    debug_printf_1("pah_8011_set_touch_setting() failed. ret = %d \n", ret);
    return ret;
}

pah_ret pah_8011_write_reg_array(const uint8_t reg_array[][2], uint32_t length)
{
    uint32_t i = 0;

    for (i = 0; i < length; ++i)
    {
        if (!pah_comm_write(reg_array[i][0], reg_array[i][1]))
        {
            debug_printf_1("pah_8011_write_reg_array(). pah_comm_write fail, i = %d \n", i);
            return pah_err_platform_fail;
        }
    }

    return pah_success;
}

pah_ret pah_8011_update_flag(void)
{
    if (!pah_comm_write(0x7F, 0x01))    //bank1
        return pah_err_platform_fail;
    if (!pah_comm_write(0x24, 0x01))
        return pah_err_platform_fail;

    return pah_success;
}

pah_ret pah_8011_device_init(void)
{
    pah_ret ret = pah_err_unknown;
    
    if (!pah_comm_write(0x7F, 0x04))    //bank4
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }
    if (!pah_comm_write(0x15, 0x69))    // set int pins as output mode
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }
    if (!pah_comm_write(0x34, 0x01))    // disable power on reset
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }

    // set int pins floating
    if (pah_comm_bus_spi == pah_comm_get_bus_type())
    {
        if (!pah_comm_write(0x2B, 0xFF))
        {
            ret = pah_err_platform_fail;
            goto FAIL;
        }
        if (!pah_comm_write(0x70, 0x08))
        {
            ret = pah_err_platform_fail;
            goto FAIL;
        }
    }
    else // i2c
    {
        if (!pah_comm_write(0x2B, 0xFE))
        {
            ret = pah_err_platform_fail;
            goto FAIL;
        }
        if (!pah_comm_write(0x70, 0x18))
        {
            ret = pah_err_platform_fail;
            goto FAIL;
        }
    }

    return pah_success;

FAIL:
    debug_printf_1("pah_8011_write_init_setting() failed. ret = %d \n", ret);
    return ret;
}

pah_ret pah_8011_startup(void)
{
    if (!pah_comm_write(0x7F, 0x04))    //bank4
        return pah_err_platform_fail;
    if (!pah_comm_write(0x69, 0x00))
        return pah_err_platform_fail;

    // When the device wakes up from power down mode, delay is necessary for the internal controller to reset the whole system and then the device is ready for the upcoming operations.
    // Without delay, the driver would be unable to write settings correctly.
    delay_ms(2);

    return pah_success;
}

pah_ret pah_8011_shutdown(pah_powerdown_int_status_e powerdown_int_status)
{
    pah_ret ret = pah_err_unknown;

    if (!pah_comm_write(0x7F, 0x04))    //bank4
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }

    if (pah_powerdown_int_status_down == powerdown_int_status)
    {
        // set int pins pull down
        if (pah_comm_bus_spi == pah_comm_get_bus_type())
        {
            if (!pah_comm_write(0x2B, 0xF9))
            {
                ret = pah_err_platform_fail;
                goto FAIL;
            }
        }
        else // i2c
        {
            if (!pah_comm_write(0x2B, 0xF8))
            {
                ret = pah_err_platform_fail;
                goto FAIL;
            }
        }
    }

    // powerdown
    if (!pah_comm_write(0x69, 0x01))
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }

    return pah_success;

FAIL:
    debug_printf_1("pah_8011_shutdown() failed. ret = %d \n", ret);
    return ret;
}

pah_ret pah_8011_read_device_package(device_package_e *device_package)
{
    pah_ret ret = pah_err_unknown;
    uint8_t data = 0;

    if (!device_package)
    {
        ret = pah_err_invalid_program;
        goto FAIL;
    }

    if (!pah_comm_write(0x7F, 0x02))    //bank2
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }
    if (!pah_comm_read(0x4A, &data))
    {
        ret = pah_err_platform_fail;
        goto FAIL;
    }
    
    data = PAH_CHECK_BIT(data, 0);
    
    *device_package = (device_package_e)data;

    return pah_success;

FAIL:
    debug_printf_1("pah_8011_read_device_package() failed. ret = %d \n", ret);
    return ret;
}
