#ifndef __pixart_heart_rate_h__
#define __pixart_heart_rate_h__


#include "pah8series_data_c.h"

#include <vector>
#include <deque>
#include <stdint.h>


namespace pixart {
    
    
    struct pah_hr_data;
    
    class heart_rate {
    public:
        typedef void (*DEBUG_PRINT_HANDLE)(const char *fmt, ...);
        typedef uint64_t (*TIME_TO_MILLISEC_HANDLE)(uint64_t time);
    
        enum gravity_mode {
            gravity_2g  = 2,
            gravity_4g  = 4,
            gravity_8g  = 8,
            gravity_16g = 16,
        };
        
        enum ret_msg {
            ret_not_ready,
            ret_success,
            ret_no_touch,
            ret_signal_poor,
            ret_fail,
        };
        
        struct ppg_sample {
            int32_t     ch1;
            int32_t     ch2;
            bool        is_touched;
            uint64_t    timestamp;
        };
        
        struct accel_sample {
            float       x;
            float       y;
            float       z;
            uint64_t    timestamp;
        };
        
        struct result {
            ret_msg ret;
            float   hr;
            int     hr_trust_level;  
        };
    
    public:
        heart_rate(TIME_TO_MILLISEC_HANDLE time_to_millisec_handler, size_t ppg_max_sample_num = 40, size_t accel_max_sample_num = 100);
        ~heart_rate();
        
        void    enable_debug_print(DEBUG_PRINT_HANDLE handler);
        
        bool    init(gravity_mode gravity = gravity_4g);
        
        void    reset();
        
        bool    set_ppg_sample(const ppg_sample &ppg_sample);
        bool    set_accel_sample(const accel_sample &accel_sample);
        
        const result&   get_result() const;
        
        
    private:
        static int gravity_to_hr_alg_gsensor_mode(gravity_mode gravity);
        static int16_t gravity_to_int16(gravity_mode gravity, float value);
    
        static const float GRAVITY_OF_EARTH;
        static const size_t PPG_SYNC_COUNT;
        static const size_t ACCEL_AXIS_NUM;
    
    private:
        void flush();
        void log_pah8series_data_header(uint32_t ch_num, uint32_t ir_ch_num, uint32_t g_sensor_mode);
        void log_pah8series_data(const pah_hr_data *pxialg_data);
    
        const size_t                PPG_MAX_SAMPLE_NUM;
        const size_t                PPG_CH_NUM;
        const size_t                ACCEL_MAX_SAMPLE_NUM;
        
        DEBUG_PRINT_HANDLE          m_debug_print_handler;
        TIME_TO_MILLISEC_HANDLE     m_time_to_millisec_handler;
        void                        *m_alg_buffer;
        gravity_mode                m_gravity_mode;
        
        std::deque<ppg_sample>      m_ppg_sample_queue;
        std::deque<accel_sample>    m_accel_sample_queue;
        
        uint64_t                    m_last_report_timestamp;
        uint8_t                     m_frame_count;
        
        result                      m_result;

    };


}


#endif // header guard
