#include "mbed.h"
#include "mDot.h"
#include "ChannelPlans.h"
#include "MTSLog.h"
#include "MTSText.h"


// Set GPIO1 for AnalogIn (used to measure analog voltage)
AnalogIn tVin(GPIO1);

 
// Create variables with settings you want.
static std::string network_name = "MultiTech";
static std::string network_passphrase = "MultiTech";
static uint8_t frequency_sub_band = 1;
static lora::NetworkType network_type = lora::PUBLIC_LORAWAN;
static uint8_t join_delay = 5;
static uint8_t ack = 0;
static bool adr = true;


// mDot Object
mDot* dot = NULL;
// Channel Plan object
lora::ChannelPlan* plan = NULL;


// Setup the debug port
Serial pc(USBTX, USBRX);


void join_network() {
    int32_t j_attempts = 0;
    int32_t ret = mDot::MDOT_ERROR;
    
    // attempt to join the network
    while (ret != mDot::MDOT_OK) {
        logInfo("attempt %d to join network", ++j_attempts);
        logInfo("do Join attempt");
        ret = dot->joinNetwork();
        logInfo("Join attempt done");
        if (ret != mDot::MDOT_OK) {
            logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
            // in some frequency bands we need to wait until another channel is available before transmitting again
            uint32_t delay_s = (dot->getNextTxMs() / 1000) + 1;
            if (delay_s < 5) {
                logInfo("waiting %lu s until next free channel", delay_s);
                wait(delay_s);
            } else {
                logInfo("sleeping %lu s until next free channel", delay_s);
                dot->sleep(delay_s, mDot::RTC_ALARM, false);
            }
        }
    }
}

void wait_to_send() {
    while(!mDot::getInstance()->getIsIdle() || 
          mDot::getInstance()->getNextTxMs() != 0 ) {

        if (mDot::getInstance()->getNextTxMs() > 1000)
            wait_ms(mDot::getInstance()->getNextTxMs());
        else
            wait_ms(1000);
    }
}


// Function to handle LoRa transmissions
void send_packet(std::vector<uint8_t> vData) {
    // join network if not joined
    if (!dot->getNetworkJoinStatus()) {
        logInfo("Joining network");
        join_network();
        logInfo("Send Blank Packet");
        std::vector<uint8_t> tx_blank;
        wait_to_send();
        dot->send(tx_blank);
    }   
    wait_to_send();
    dot->send(vData);   
}

void send_string_packet(std::string sendStr)
{
    std::vector<uint8_t> tx_data;
    for (std::string::iterator it = sendStr.begin(); it != sendStr.end(); it++)
        tx_data.push_back((uint8_t) *it);
    send_packet(tx_data);
}

int main() {
    // Set baud rate for debug port.
    pc.baud(115200);
    
    mts::MTSLog::setLogLevel(mts::MTSLog::INFO_LEVEL);

    // Create new channel plan object
    plan = new lora::ChannelPlan_US915();
    assert(plan);

    // Create a mDot object that uses that plan
    dot = mDot::getInstance(plan);
    assert(dot);

    // You can now start calling mDot functions using that mDot object dot->function()

    // Default settings and session to start from a well-known state
    logInfo("defaulting Dot configuration");
    dot->resetConfig();
    dot->resetNetworkSession();

    // make sure library logging is turned on
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    // Configure settings

    // Configure the Join mode to OTA
    if (dot->setJoinMode(mDot::OTA) != mDot::MDOT_OK) {
        logError("failed to set network join mode to OTA");
    }

    // Configure Network Name. If using a Network ID (App-EUI) then use setNetworkId(network_id) instead.
    if (dot->setNetworkName(network_name) != mDot::MDOT_OK) {
        logError("failed to set network name to \"%s\"", network_name.c_str());
    }

    // Configure Network Passphrase. If using a Network Key (App-Key) use setNetworkKey(network_key) instead.
    if (dot->setNetworkPassphrase(network_passphrase) != mDot::MDOT_OK) {
        logError("failed to set network passphrase to \"%s\"", network_passphrase.c_str());
    }

    if (dot->setFrequencySubBand(frequency_sub_band) != mDot::MDOT_OK) {
        logError("failed to set frequency sub band to %u", frequency_sub_band);
    }

    if (dot->setPublicNetwork(network_type) != mDot::MDOT_OK) {
        logError("failed to set network type");
    }

    if (dot->setAck(ack) != mDot::MDOT_OK) {
        logError("failed to set acks to %u", ack);
    }

    // enable or disable Adaptive Data Rate
    dot->setAdr(adr);

    // Configure the join delay
    dot->setJoinDelay(join_delay);


    // save changes to configuration
    logInfo("saving configuration");
    if (!dot->saveConfig()) {
        logError("failed to save configuration");
    }
    
    
    int sleep_secs = 10;
    while (true) {
        std::vector<uint8_t> tx_data;
        
        // Read AnalogIn value and calculate voltage.
        // Max read_u16() value is 65535, and reference voltage is v3.0
        // voltage = (read_u16() /65535) * 3.0
        float Vin_Volt = ((float)tVin.read_u16()/65535.0) * 3.0;
        logInfo("Vin : %u, Voltage: %f", tVin.read_u16(), Vin_Volt);
        
        // Send Raw AnalogIn Read as 2 bytes
        uint16_t RawData = tVin.read_u16();
        tx_data.push_back((RawData >> 8) & 0xFF);
        tx_data.push_back(RawData & 0xFF);
        logInfo("Sending Raw AnalogIn Data: [0x%04X]", RawData);
        send_packet(tx_data);
                
        tx_data.clear();
        // Send Float as Bytes
        unsigned long d = *(unsigned long *)&Vin_Volt;
        tx_data.push_back((d >> 24) & 0xFF);
        tx_data.push_back((d >> 16) & 0xFF);
        tx_data.push_back((d >> 8) & 0xFF);
        tx_data.push_back(d & 0xFF);
        logInfo("Sending Float as 4 bytes: [0x%08X]", d);
        send_packet(tx_data);
            
        
        // Send Vin as a readable string
        std::string strVin = std::to_string(Vin_Volt);
        logInfo("Sending Voltage as String: %s", strVin.c_str());
        send_string_packet(strVin);
        
                        
        // sleep
        // Wait for mDot to become idle before sleeping
        while (true) {
            if (dot->getIsIdle()) {
                break;
            }
            wait_ms(10);
        }
        logInfo("Sleeping for %d seconds.", sleep_secs);
        dot->sleep(10, mDot::RTC_ALARM, false); // Deepsleep not currently supported on mDot
    }

    return 0;
}
