/*==============================================================================
  BC127 Melody mbed Echo App
  Copyright (c) 2013 BlueCreation, MIT License
  
  This is a demo application for the BC127 module available from BlueCreation
  and and the mbed.
  You will need an mbed, a BC127, and a phone that support Bluetooth and SPP and
  has the Melody application installed. Application is available on the Play
  Store here:
  
  Connect and power the BC127 module according to the Manual and Data sheet.
  Connect Power, Gnd, and connect TX on the module to Pin 9 on the mbed and 
  Rx to pin 10. Also connect ground from the module to the mbed.
  
  Getting this to work:
  
  1) Compile and load the BC127 Melody mbed Echo App.
  2) Reset the mbed to load the app.
  3) Reset or power up the BC127.
  You will see LED1 on the mbed light up.
  The BC127 is now discoverable.
  4)Find BC127 using your phone and pair to it.
  The device name will be of the type BlueCreation-XXXXXX.
  5) Open the Melody application and connect to the BC127.
  LED2 on the mbed should light up in addition to LED1. You are now connected!
  When LED3 lights up the mbed has configured the BC127.
  6) Follow the instructions you just received in the Melody app!
  
  Note: If you reset the BC127 you will need to reset the mbed to enable it to
  connect and configure the BC127.
  If you just disconnect and reconnect the phone, you will be sent back the SPP
  open indicator event coming from the BC127.
  
*******************************************************************************/
 
 
#include "mbed.h"
#include "Serial.h"
#include "string.h"
 
#define SIZE 13
char UARTBuff[SIZE];
 
/* Responses from Melody, in order characters are stored in FIFO */
char SppConn[SIZE] = {'\r','\n','P','P','S',' ','K','O','_','N','E','P','O'};
char Ok[4] = {'\r','\n','K','O'};
 
/* Use LED to indicate mbed state */
DigitalOut myled1(LED1); //started parsing
DigitalOut myled2(LED2); // connected
DigitalOut myled3(LED3); // configured and in echo mode
DigitalOut myled4(LED4); // toggles for every character received in echo mode
 
/* track state internally with these flags */
int connected = 0;
int loopback = 0;
 
/* Create a serial instance using pins 9 and 10 for mbed tx and rx */
Serial melody(p9,p10);
 
 
/*              -=Interrupt handler for Serial Rx interrupt=-
 
    This function is called when an interrupt is generated on the UART RX.
    This happens for every character received.
 
    This function stores received characters in a buffer, compares it to known 
    responses and passes commands to the BC127 Melody module based on the 
    received responses.
    
    Once it configures the BC127 it stops storing characters and goes into echo
    (loopback) mode and echoes back any character received.
 
*/
 
void RxParse(void)
{
    if(!loopback) {
        /* A character has been received and we are not in loopback.
           Starting to look for a SPP connection indication          */
 
        myled1 = 1; //indicate we have started parsing
 
        /* Shifts characters in the FIFO buffer.
           Note: a circular buffer would have been much faster, however,
           we would not have been able to use memcmp() so it is a good
           trade-off in terms of speed.                                */
 
        for(int i = SIZE-2; i>0; i--) 
        {
            UARTBuff[i+1] = UARTBuff[i];
        }
        UARTBuff[1] = UARTBuff[0];
        UARTBuff[0] = melody.getc(); // store current character
 
        if(!connected) { // SPP connection has not been detected yet
            /* Check buffer for SPP connection indication */
            if(memcmp(&UARTBuff[0],&SppConn[0],SIZE)==0)
             {
                /* SPP connection detected */
                myled2 = 1; // indicate to the world
                melody.printf("ENTER_DATA\r\n"); // Configure BC127
                connected = 1; // change internal state
 
                return;
            }
            return;
 
        }
        /* We are connected and waiting for a confirmation that our
           configuration has been accepted and the BC127 has entered
           DATA_MODE                                                  */
        else if(memcmp(&UARTBuff[0],&Ok[0],4) == 0) 
        {
            /* Success! We have configured the BC127 and we can now
               send some instruction to Bluetooth terminal via BC127 */
 
            myled3 = 1; /* indicate state to the world */
            melody.printf("\r\n\r\n");
            melody.printf("*********************************\n\r");
            melody.printf("* Melody mbed Cookbook Echo App *\r\n");
            melody.printf("* (c) BlueCreation 2013         *\r\n");
            melody.printf("*********************************\n\r");
            melody.printf("\r\n");
            melody.printf(">> mbed now in SPP Echo mode.\r\n");
            melody.printf(">> Type any character:\r\n>> ");
            loopback = 1; // change internal state
 
            return;
        }
    } 
    else
    {
        /* We are in loopback mode. Get character received from UART and
       send it back out indicating we have echoed it              */
        melody.printf("\n\r<< mbed echo: %c\n\r>> ", melody.getc());
        myled4 = !myled4; // toggle LED4 for each character received
        return;
    }
 
    return;
}
 
int main()
{
    /* The Melody firmware on the BC127 runs at 9600kpbs, 8 data bits,
       no parity bits, 1 stop bit and no HW flow control.
       These are the default serial settings for mbed, so we do not
       need to set those on the serial interface.
 
       We just need to handle incoming characters.
       Therefore we attach and interrupt handler to the Rx interrupt.
       The mbed firmware will then call our RxParse function every
       time a character is received on the serial interface         */
 
    melody.attach(RxParse);
 
}
