#ifndef __SRAM23X__H_
#define __SRAM23X__H_

// Includes
#include <string> 

#include "mbed.h"

// Example
/*
*/

// Defines
#define SRAM23X_READ        0x03
#define SRAM23X_WRITE       0x02
#define SRAM23X_RDSR        0x05
#define SRAM23X_WRSR        0x01
#define SRAM23X_SEQ         0x40

#define SRAM23X_NoError     0x00
#define SRAM23X_OutOfRange  0x01
#define SRAM23X_CsError     0x02
#define SRAM23X_MallocError 0x03

#define SRAM23X_MaxError       4

static std::string _ErrorMessageSRAM23X[SRAM23X_MaxError] = {
                                                          "",
                                                          "Data address out of range"
                                                          "Chip select error",
                                                          "Memory allocation error"       
                                                        };

// Class
class SRAM23X {
public:
    enum TypeSram {S23X640=8192,S23X256=32768} Type;
    
    /*
     * Constructor, initialize the sram on spi interface.
     * @param mosi : mosi spi pin (PinName)
     * @param miso : miso spi pin (PinName)
     * @param sclk : sclk spi pin (PinName)
     * @param cs   : chip select pin (PinName)
     * @param type : sram type (TypeSram) 
     * @return none
    */
    SRAM23X(PinName mosi, PinName miso, PinName sclk, PinName cs, TypeSram type);
    
   /*
     * Sequential read bytes
     * @param address : start address (uint16_t)
     * @param data : bytes array to read (int8_t *)
     * @param size : number of bytes to read (uint16_t)
     * @return none
    */
    void read(uint16_t address, int8_t *data, uint16_t size);
    
    /*
     * Sequential read anything
     * @param address : start address (uint16_t)
     * @param data : data array to read (void *)
     * @param size : number of bytes to read (uint16_t)
     * @return none
    */
    void read(uint16_t address, void *data, uint16_t size);
    
    /*
     * Sequential read int16
     * @param address : start address (uint16_t)
     * @param data : short array to read (int16_t *)
     * @param size : number of bytes to read (uint16_t)
     * @return none
    */
    void read(uint16_t address, int16_t *data, uint16_t size);
    
    /*
     * Sequential read int32
     * @param address : start address (uint16_t)
     * @param data : int array to read (int32_t *)
     * @param size : number of bytes to read (uint16_t)
     * @return none
    */
    void read(uint16_t address, int32_t *data, uint16_t size);
    
    /*
     * Sequential read float
     * @param address : start address (uint16_t)
     * @param data : float array to read (float *)
     * @param size : number of bytes to read (uint16_t)
     * @return none
    */
    void read(uint16_t address, float *data, uint16_t size);
    
    /*
     * Sequential write bytes
     * @param address : start address(uint16_t)
     * @param data : bytes array to write (int8_t *)
     * @param size : number of bytes to write (uint16_t)
     * @return none
    */
    void write(uint16_t address, int8_t *data, uint16_t size);
    
    /*
     * Sequential write anything
     * @param address : start address (uint16_t)
     * @param data : data array to write (void *)
     * @param size : number of bytes to write (uint16_t)
     * @return none
    */
    void write(uint16_t address, void *data, uint16_t size);
    
    /*
     * Sequential write int16
     * @param address : start address (uint16_t)
     * @param data : short array to write  (int8_t *)
     * @param size : number of bytes to write (uint16_t)
     * @return none
    */
    void write(uint16_t address, int16_t *data, uint16_t size);
    
    /*
     * Sequential write int32
     * @param address : address to write (uint16_t)
     * @param data : int array to write (int32_t *)
     * @param size : number of bytes to write (uint16_t)
     * @return none
    */
    void write(uint16_t address, int32_t *data, uint16_t size);
    
    /*
     * Sequential write float
     * @param address : address to write (uint16_t)
     * @param data : float array to write (float *)
     * @param size : number of bytes to write (uint16_t)
     * @return none
    */
    void write(uint16_t address, float *data, uint16_t size);
    
    /*
     * Get sram size in bytes
     * @param : none
     * @return size in bytes (uint16_t)
    */
    uint16_t getSize(void);
    
    /*
     * Select chip
     * @param : none
     * @return none
    */
    void select(void);
    
    /*
     * Deselect chip
     * @param : none
     * @return none
    */
    void deselect(void);
    
    /*
     * Read status register
     * @param : none
     * @return status register  value (uint8_t)
    */
    uint8_t readStatus(void);
    
    /*
     * Write status register
     * @param : status value (uint8_t)
     * @return none
    */
    void writeStatus(uint8_t status);
    
    /*
     * Get current error message
     * @param  : none
     * @return current error message(std::string)
    */
    std::string getErrorMessage(void)
    { 
      return(_ErrorMessageSRAM23X[_errnum]);
    }
    
    /*
     * Get the current error number (SRAM23X_NoError if no error)
     * @param  : none
     * @return current error number (uint8_t)
    */
    uint8_t getError(void);
    
//---------- local variables ----------
private:
    SPI _spi;             // Local spi communication interface instance
    DigitalOut *_cs;      // Chip select
    uint8_t _errnum;      // Error number
    TypeSram _type;       // SRAM type
    uint16_t _size;       // Size in bytes
    bool checkAddress(uint16_t address); // Check address range
//-------------------------------------

    /*
     * Initialization
     * @param address : address to write (uint16_t)
     * @param size : size to write in bytes (uint16_t)
     * @param mode : read or write (uint8_t)
     * @return error code, 0 if no error
    */
    uint8_t init(uint16_t address, uint16_t size, uint8_t mode);
};
#endif