/***********************************************************
Author: Bernard Borredon
Date: 27 december 2011
Version: 1.0
************************************************************/
#include "sram23x.h"

#define HIGH(x) (x >> 8)
#define LOW(x) (x & 0xff)

SRAM23X::SRAM23X(PinName mosi, PinName miso, PinName sclk, PinName cs, TypeSram type) : _spi(mosi, miso, sclk) 
{
  
  _errnum = SRAM23X_NoError;
  _type = type;
  _cs = new DigitalOut(cs);
  if(_cs == NULL)
    _errnum = SRAM23X_CsError;
    
  // Don't select chip
  deselect();
  
    // Size in bytes
  _size = _type;
  
  // Set SPI frequency
  _spi.frequency(10000000);
  
  // Set format
  _spi.format(8,0);
  
  // Mode sequential
  select();
  writeStatus(SRAM23X_SEQ);
  deselect();
}

// Initialization
uint8_t SRAM23X::init(uint16_t address, uint16_t size, uint8_t mode)
{
  
  // Check error
  if(_errnum) 
    return(_errnum);
    
  // Check address
  if(!checkAddress(address)) {
    _errnum = SRAM23X_OutOfRange;
    return(_errnum);
  }
  
  // Check address
  if(!checkAddress(address + size - 1)) {
    _errnum = SRAM23X_OutOfRange;
    return(_errnum);
  }
  
  // No error
  _errnum = SRAM23X_NoError;

  select();
  // Acces instruction
  _spi.write(mode);
  
  // High address part
  _spi.write(HIGH(address));
  
  // Low address part
  _spi.write(LOW(address));
  
  return(_errnum);
}

// Sequential read bytes
void SRAM23X::read(uint16_t address, int8_t *data, uint16_t size)
{
  uint8_t ret;
  uint16_t i;
  
  ret = init(address,size,SRAM23X_READ);
  if(ret != 0) {
    // Deselect chip
    deselect();
    return;
  }
  
  // Datas
  for(i = 0;i < size;i++) {
    data[i] = _spi.write(0);
  }
  
  // Deselect chip
  deselect();
  
}

// Sequential read anything
void SRAM23X::read(uint16_t address, void *data, uint16_t size)
{
  int8_t *cmd = NULL;
  
  cmd = (int8_t *)malloc(size);
  if(cmd == NULL) {
    _errnum = SRAM23X_MallocError;
    return;
  }
  
  read(address,cmd,size);
  
  memcpy(data,cmd,size);
  
  free(cmd);
}

// Sequential read int16
void SRAM23X::read(uint16_t address, int16_t *data, uint16_t size)
{
  uint16_t i,j;
  uint8_t cmd[2];
  uint8_t ret;
  
  ret = init(address,size,SRAM23X_READ);
  if(ret != 0) {
    // Deselect chip
    deselect();
    return;
  }
  
  // Datas
  for(i = 0;i < size / 2;i = i + 2) {
     for(j = 0;j < 2;j++)
        cmd[j] = _spi.write(0);
     memcpy(&data[i],cmd,2);
  }
  
  // Deselect chip
  deselect();

}

// Sequential read int32
void SRAM23X::read(uint16_t address, int32_t *data, uint16_t size)
{
  uint16_t i,j;
  uint8_t cmd[4];
  uint8_t ret;
  
  ret = init(address,size,SRAM23X_READ);
  if(ret != 0) {
    // Deselect chip
    deselect();
    return;
  }
  
  // Datas
  for(i = 0;i < size / 4;i = i + 4) {
     for(j = 0;j < 4;j++)
        cmd[j] = _spi.write(0);
     memcpy(&data[i],cmd,4);
  }
  
  // Deselect chip
  deselect();

}

// Sequential read float
void SRAM23X::read(uint16_t address, float *data, uint16_t size)
{
  uint16_t i,j;
  uint8_t cmd[4];
  uint8_t ret;
  
  ret = init(address,size,SRAM23X_READ);
  if(ret != 0) {
    // Deselect chip
    deselect();
    return;
  }
  
  // Datas
  for(i = 0;i < size / 4;i = i + 4) {
     for(j = 0;j < 4;j++)
        cmd[j] = _spi.write(0);
     memcpy(&data[i],cmd,4);
  }
  
  // Deselect chip
  deselect();

}

// Sequential write bytes
void SRAM23X::write(uint16_t address, int8_t *data, uint16_t size)
{
  uint16_t i;
  uint8_t ret;
  
  ret = init(address,size,SRAM23X_WRITE);
  if(ret != 0) {
    // Deselect chip
    deselect();
    return;
  }
  
  // Datas
  for(i = 0;i < size;i++) {
    _spi.write(data[i]);
  }
  
  // Deselect chip
  deselect();
  
}

// Sequential write anything
void SRAM23X::write(uint16_t address, void *data, uint16_t size)
{
  int8_t *cmd = NULL;
  
  cmd = (int8_t *)malloc(size);
  if(cmd == NULL) {
    _errnum = SRAM23X_MallocError;
    return;
  }
  
  memcpy(cmd,data,size);
  
  write(address,cmd,size);
  
  free(cmd);
}

// Sequential write int16
void SRAM23X::write(uint16_t address, int16_t *data, uint16_t size)
{
  uint16_t i,j;
  uint8_t ret;
  uint8_t cmd[2];
  
  ret = init(address,size,SRAM23X_WRITE);
  if(ret != 0) {
    // Deselect chip
    deselect();
    return;
  }
  
  // Datas
  for(i = 0;i < size / 2;i = i + 2) {
     memcpy(cmd,&data[i],2);
     for(j = 0;j < 2;j++)
        _spi.write(cmd[j]);
  }
  
  // Deselect chip
  deselect();
  
}

// Sequential write int32
void SRAM23X::write(uint16_t address, int32_t *data, uint16_t size)
{
  uint16_t i,j;
  uint8_t ret;
  uint8_t cmd[4];
  
  ret = init(address,size,SRAM23X_WRITE);
  if(ret != 0) {
    // Deselect chip
    deselect();
    return;
  }
  
  // Datas
  for(i = 0;i < size / 4;i = i + 4) {
     memcpy(cmd,&data[i],4);
     for(j = 0;j < 4;j++)
        _spi.write(cmd[j]);
  }
  
  // Deselect chip
  deselect();
  
}

// Sequential write float
void SRAM23X::write(uint16_t address, float *data, uint16_t size)
{
  uint16_t i,j;
  uint8_t ret;
  uint8_t cmd[4];
  
  ret = init(address,size,SRAM23X_WRITE);
  if(ret != 0) {
    // Deselect chip
    deselect();
    return;
  }
  
  // Datas
  for(i = 0;i < size / 4;i = i + 4) {
     memcpy(cmd,&data[i],4);
     for(j = 0;j < 4;j++)
        _spi.write(cmd[j]);
  }
  
  // Deselect chip
  deselect();
  
}

// Get SRAM size in bytes
uint16_t SRAM23X::getSize(void)
{
  return(_size);
}

// Select chip
void SRAM23X::select(void)
{
  _cs->write(0);
  wait_us(10.0);
}

// Deselect chip
void SRAM23X::deselect(void)
{

  wait_us(10.0);
  _cs->write(1);
}

// Read status register
uint8_t SRAM23X::readStatus(void)
{
  uint8_t ret;
  
  select();
  _spi.write(SRAM23X_RDSR);
  ret = _spi.write(0);
  deselect();
  
  return(ret);
}

// Write status register
void SRAM23X::writeStatus(uint8_t status)
{
  
  select();
  _spi.write(SRAM23X_WRSR);
  _spi.write(status);
  deselect();
}

// Check address
bool SRAM23X::checkAddress(uint16_t address)
{
  bool ret = true;
  
  if(address >= _type)
    ret = false;
  
  return(ret);
}  

// Get current error number
uint8_t SRAM23X::getError(void)
{ 
  return(_errnum);
}