#ifndef __Led_H__
#define __Led_H__

/***********************************************************
Author: Bernard Borredon
Date : 24 decembre 2015
Version: 1.0
  - creation.
************************************************************/

#include "mbed.h"

/** Led class
 *  Used to drive a led : on, on for a duration, off, toggle, toggle with a period.
 *
 * @code
 * #include "Led.h"
 *
 * Led led1(LED1);  // LPC1768 LED1
 * Led led2(LED2);  // LPC1768 LED2
 * Led led3(LED3);  // LPC1768 LED3
 * Led led4(LED4);  // LPC1768 LED4
 *
 * int main()
 * {
 *   int32_t count = 0;
  
 *   // Light on LED1
 *   led1.on();
 * 
 *   // Toggle LED2 with a 2000ms period
 *   led2.toggle(2000);
 *
 *   // Toggle LED3 with a 4000ms period
 *   led3.toggle(4000,true);
 * 
 *  while(1) {
 *             // Pulse LED4 for 1000ms each 10s
 *              led4.pulse(1000);
 *              count++;
 *              wait(10.0);
 *              
 *              // After 60s stop toggle LED2
 *              if(count == 6)
 *                led2.toggle(0,false);
 *   }
 *   return(0);
 * }
 * @endcode
 */

/** LedBase class
 *  Used to drive a led : on, off, toggle.
 *
 */

class LedBase {
    public :
        /** Class constructor : create and initialize LedBase instance
         * @param pin : name of the pin connected to the led (PinName)
         * @returns none
         */
        LedBase(PinName pin);
        
        /** Toggle led 
         * @param none
         * @returns none
         */
        void toggle(void);
        
        /** Light on led
         * @param none
         * @returns none
         */
        void on(void);
        
        /** Light off led
         * @param none
         * @returns none
         */
        void off(void);
        
        /** Get led status (on or off)
         * @param none
         * @returns led status (bool)
         *      false : led is off
         *      ture  : led is on
         */
        bool isOn(void);
        
    protected :
        DigitalOut _pin;
        bool _isOn;
};
 
class Led : public LedBase {
    public :
        /** Class constructor : create and initialize Led instance
         * @param pin : name of the pin connected to the led (PinName)
         * @returns none
         */
        Led(PinName pin);
         
        /** Periodically toggle led
         * @param period : toggle period in ms (uinit16_t)
         * @param mode : stop auto toggle if false (bool)
         * @returns none
         */
        void toggle(uint16_t period, bool mode = true);
        
        /** Light on led for a duration
         * @param time : light on duration in ms (uint16_t)
         * @returns none
         */
        virtual void pulse(uint16_t time);
    
    private :
        LedBase *_led;
        Timeout _tPulse;
        Ticker _tToggle;
        void atTimer(void);
        void toggleCB(void);
};

#endif
