#ifndef DS1621__H_
#define DS1621__H_

// Includes
#include <string>

#include "mbed.h"

// Example
/*
#include <string>

#include "mbed.h"
#include "ds1621.h"
#define DS1621_ADDR 0     // I2c DS1621 address is 0x00

static void myerror(std::string msg)
{
  printf("Error %s\n",msg.c_str());
  exit(1);
}

int main()
{
  DS1621 ds(p9,p10,DS1621_ADDR);  // Declare DS1621 i2c with sda = p9 and scl = p10
  bool tlf,thf;
  uint8_t i;
  uint8_t config;
  uint16_t count = 0;
  float temp = 0.0;
  
  printf("Test DS1621 I2C\n\n");
  
  // Check error
  if(ds.getError() != 0)
    myerror(ds.getErrorMessage());

  // Config with OneShot and Pol high and test error
  ds.setConfig(DS1621_1Shot | DS1621_PolHigh);
  if(ds.getError() != 0)
    myerror(ds.getErrorMessage());
  
  // Read Config register
  config = ds.getConfig();
  if(ds.getError() != 0)
    myerror(ds.getErrorMessage());
  printf("DS1621 Config : %02X\n",config);
   
  printf("Test HR Temperature\n"); 
  
  // Test HR temperature
  for(i = 0;i < 24;i++) {
     temp = ds.getHrTemp();
     if(ds.getError() != 0)
       myerror(ds.getErrorMessage());
     printf("Measurment at start + %d seconds, Temperature HR = %.2f\n",(count*5),temp);
     wait(5.0);
     count++;
  }
 
  printf("Test Temperature\n"); 
 
  // Test temperature
  count = 0;
  for(i = 0;i < 12;i++) {
     temp = ds.getTemp();
     if(ds.getError() != 0)
       myerror(ds.getErrorMessage());
     printf("Measurment at start + %d seconds, Temperature = %.2f\n",(count*5),temp);
     wait(5.0);
     count++;
  }
  
  // Config with Continuous and Pol high
  ds.setConfig(DS1621_PolHigh); 
  if(ds.getError() != 0)
    myerror(ds.getErrorMessage());
  
  // Read Config register
  config = ds.getConfig();
  if(ds.getError() != 0)
    myerror(ds.getErrorMessage());
  printf("DS1621 Config : %02X\n",config);
  
  // Write trigger temperatures
  ds.setTemperature(18.0,DS1621_Access_TL);
  if(ds.getError() != 0)
    myerror(ds.getErrorMessage());
  ds.setTemperature(25.5,DS1621_Access_TH);
  if(ds.getError() != 0)
    myerror(ds.getErrorMessage());
  
  printf("Test temperature and triggers\n");
  while(1) {
             temp = ds.getTemp();
             if(ds.getError() != 0)
               myerror(ds.getErrorMessage());
             ds.getTF(tlf,thf);
             if(ds.getError() != 0)
               myerror(ds.getErrorMessage());
             printf("Temperature=%.2f  TLF %d  THF %d\n",temp,tlf,thf);
             wait(5.0);
  }
}
*/

// Defines
#define DS1621_address     0x90

#define DS1621_PolHigh     0x02
#define DS1621_1Shot       0x01
#define DS1621_DONE        0x80

#define DS1621_CFG_1SHOT      0
#define DS1621_CFG_POL        1
#define DS1621_CFG_TLF        5
#define DS1621_CFG_THF        6
#define DS1621_CFG_DONE       7

#define DS1621_Access_TH      0
#define DS1621_Access_TL      1

#define DS1621_NoError     0x00
#define DS1621_BadAddress  0x01
#define DS1621_I2cError    0x02
#define DS1621_ParamError  0x03

#define DS1621_MaxError       4

static std::string _ErrorMessageDS1621[DS1621_MaxError] = {
                                                            "",
                                                            "Bad chip address",
                                                            "I2C error (nack)",
                                                            "Invalid parameter",
                                                          };

// Class
class DS1621 {
public:
    /*
     * Constructor, initialize the ds1621 on i2c interface.
     * @param sda : sda i2c pin (PinName)
     * @param scl : scl i2c pin (PinName)
     * @param address : ds1621 address between 0 and 7 (uint8_t) 
     * @return none
    */
    DS1621(PinName sda, PinName scl, uint8_t address);

    /*
     * Get temperature from the ds1621 with 0.5 degrees resolution
     * @param : none
     * @return current temperature in degrees Celsius between -55 and +125 (float)
    */
    float getTemp(void);
    
    /*
     * Get temperature from the ds1621 with 0.01 degrees resolution
     * @param : none
     * @return current temperature in degrees Celsius between -55 and +125 (float) 
    */
    float getHrTemp(void);
    
    /*
     * Read config register
     * @param : none
     * @return  current config register value (uin8_t)
    */
    uint8_t getConfig(void);
    
    /*
     * Write config register
     * @param config : config value (uint8_t) 
     * @return none
    */
    void setConfig(uint8_t config);
    
    /*
     * Start / stop convert
     * @param flag : start convert if true else stop 
     * @return none
    */
    void startConvert(bool flag);
    
    /*
     * Wait end of conversion (Config register bit 7 at 1)
     * @param : none
     * @return none
    */
    void waitEndConvert(void);
    
    /*
     * Get Temperature High Flag
     * @param : none
     * @return thf flag (bool)
    */
    bool getTHF(void);
    
    /*
     * Get Temperature Low Flag
     * @param : none
     * @return tlf flag (bool)
    */
    bool getTLF(void);
    
    /*
     * Get Temperature Low and High Flags
     * @param tlf : temperature low flag (uint8_t&)
     * @param thf : temperature high flag (uint8_t&)
     * @return none
    */
    void getTF(bool& tlf,bool& thf);
    
     /*
    *Get 1Shot bit
    *@param : none
    *@return true if 1Shot (bool)
    */
    bool get1Shot(void);
    
    /*
    *Set Temperature (High or Low)
    *@param temp : temperature, the fractionnal part is rounded to 0.5 (float)
    *@param trig : mode,low or high (uint8_t) [DS1621_Access_TL DS1621_Access_TH]
    *@return none
    */
    void setTemperature(float temp,uint8_t trig);
    
     /*
    *Get Temperature (High or Low)
    *@param trig : mode,low or high (uint8_t) [DS1621_Access_TL DS1621_Access_TH]
    *@return  temperature, with 0.5 degrees Celsius resolution (float)
    */
    float getTemperature(uint8_t trig);
    
    /*
    *Get current error message
    *@param  : none
    *@return current error message(std::string)
    */
    std::string getErrorMessage(void)
    { 
      if(_errnum < DS1621_MaxError)
        return(_ErrorMessageDS1621[_errnum]);
      else
        return("errnum out of range");
    }
    
    /*
     * Get the current error number (DS1621_NoError if no error)
     * @param  : none
     * @return current error number (uint8_t)
    */
    uint8_t getError(void);
    
//---------- local variables ----------
private:
    I2C _i2c;        // Local i2c communication interface instance
    int _address;    // Local ds1621 i2c address
    uint8_t _errnum; // Error number
    uint8_t _config; // Config register value
//-------------------------------------
};

#endif