// Lukas Bieri, Matthias, Manuel Meier & Noa Melchior 
// based on Work by Matthias Grob & Manuel Stalder 
// ETH 2017

#ifndef _NODES_H
#define _NODES_H

#include "mbed.h"
#include "DecaWave.h"
#include "frames.h"

#include "globals.h"

#define TIMEUNITS_TO_US       (1/(128*499.2))    // conversion between the decawave timeunits (ca 15.65ps) to microseconds.
#define US_TO_TIMEUNITS       (128*499.2)        // conversion between microseconds to the decawave timeunits (ca 15.65ps).
#define MMRANGING_2POWER40     1099511627776     // decimal value of 2^40 to correct timeroverflow between timestamps
#define ADRESSES_COUNT         128               // Defines the Adress Space that is covered when Ranging for all
  
  //static int ADRESSES_COUNT=1;                                           // Adress Space: 0 - (ADRESSES_COUNT - 1)
#define MAX_TRIES              0                // Number of times a Anchor is pinged until determined it is not in range
#define BASE_STATION_ADDR      15                // Defines the Adress of the Base Station (reserved Adress)

// Constants for Base_Orders
#define RANGE_ALL               1                
#define RANGE_ONE               0           
#define NOT_USED                0xff

// Constant for Switch Types Orders
#define BECOME_BEACON           1                
#define BECOME_ANCHOR           0


class Node {
    public:
        Node(DecaWave& DW);
    
        void setAnchor(bool anc);
        bool isAnchor();  
        bool isBaseStation();      
        void setAddress(uint8_t adr);
        uint8_t getAddress(); 
        
        virtual uint8* getRecFrameRef() = 0;         
        virtual uint16 getRecFrameLength() = 0;
        
        virtual void callbackRX(uint64_t RxTime) = 0;
        virtual void callbackTX(uint64_t TxTime) = 0; 
        
        //Frametype of the form 1001XXXX. First 4 Bits for uniqueness
        enum FrameType{
            PING = 0x91,
            ANCHOR_RESPONSE = 0x92,
            BEACON_RESPONSE = 0x93,
            TRANSFER_FRAME = 0x94,
            DISTANCES_FRAME = 0x95,
            STREAM_TO_BASE = 0x96,
            BASE_ORDER = 0x97,
            SWITCH_TYPE = 0x98,  
            BASE_ORDER_ACK = 0x99         
        };              
        
    protected:
        DecaWave& dw;    
        uint8_t address; 
        bool Anchor;                 
};

class BaseStationNode : public Node {
    public:
        BaseStationNode(DecaWave& DW);
    
        virtual uint8* getRecFrameRef();         
        virtual uint16 getRecFrameLength();
            
        virtual void callbackRX(uint64_t RxTime);
        virtual void callbackTX(uint64_t TxTime);  
        
        void sendOrder(uint8_t beacon_destination, uint8_t anchor_destination, uint8_t action, uint16_t repetitions, uint8_t type);
        
    private:     
        Timer LocalTimer;   
        bool ack;    
        RangingFrame rangingFrame;                  // buffer in class for sending a frame (not made locally because then we can recall in the interrupt what was sent)
    public:    
        StreamFrame receivedStreamFrame; 
};

class AnchorNode : public Node {
    public:    
        AnchorNode(DecaWave& DW);
        
        virtual uint8* getRecFrameRef();
        virtual uint16 getRecFrameLength();  
         
        virtual void callbackRX(uint64_t RxTime);
        virtual void callbackTX(uint64_t TxTime);    
         
    private:
        uint64_t receiverTimestamps[ADRESSES_COUNT][3];  

        void sendAnswer(uint8_t destination, uint8_t type);
        void sendTransferFrame(uint8_t destination, int timestamp); 
        void sendBaseAck();   

        void correctReceiverTimestamps(uint8_t source);

        RangingFrame rangingFrame;                  // buffer in class for sending a frame (not made locally because then we can recall in the interrupt what was sent)
        
    public:    
        ExtendedRangingFrame receivedFrame;   
};



class BeaconNode : public Node {
    public:
        BeaconNode(DecaWave& DW);
    
        virtual void requestRanging(uint8_t destination);
        virtual void requestRangingAll();
     
        float getDistance(uint8_t index);  
        float getSignalStrength(uint8_t index);         
        int getMode();
                  
        virtual uint8* getRecFrameRef();
        virtual uint16 getRecFrameLength();
        
        virtual void callbackRX(uint64_t RxTime);
        virtual void callbackTX(uint64_t TxTime); 
        
        uint16_t getRepetitions();
        void decreaseRepetitions();
        uint8_t getDestination();
        void clearRec();
        
         
    private:  
        Timer LocalTimer;
        float distances[ADRESSES_COUNT];          // Array containing the finally calculated Distances to the anchors 
        float signalStrength[ADRESSES_COUNT];          // Array containing the finally calculated Distances to the anchors 
        uint64_t senderTimestamps[ADRESSES_COUNT][3];
        volatile bool acknowledgement[3];                                // flag to indicate if ranging has started (0) and succeeded (1)
        int32_t tofs[ADRESSES_COUNT];                          // Array containing time of flights for each node (index is address of node)
        int8_t noRec[ADRESSES_COUNT];
    
        void sendPingFrame(uint8_t destination);
        void sendAnswer(uint8_t destination, uint8_t type);
        void sendStreamFrame(uint8_t anchor_addr);
        void sendBaseAck();
        
        inline float calibratedDistance(uint8_t destination);
        void correctSenderTimestamps(uint8_t source);
       
        RangingFrame rangingFrame;                  // buffer in class for sending a frame (not made locally because then we can recall in the interrupt what was sent)
    
        uint8_t mode;      
        uint8_t destination;   
        uint16_t repetitions;    
    
    public:    
        ExtendedRangingFrame receivedFrame;    
    
    
};

#endif

