/*-
 * BSD 2-Clause License
 *
 * Copyright (c) 2012-2018, Jan Breuer
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @file   scpi-def.c
 * @date   Thu Nov 15 10:58:45 UTC 2012
 *
 * @brief  SCPI parser test
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "uart_mbed.h"
#include "main_init.h"
#include "i2c_mbed_fpga.h"
#include "scpi/scpi.h"
#include "scpi/parser.h"
#include "scpi-def.h"

extern i2c_mbed_fpga i2c;

static scpi_result_t set_duty(scpi_t * context) {
    int input = 0;
    int duty_cycle_scaled = 0;
  
    /* read first parameter if present */
    if (!SCPI_ParamInt(context, &input, TRUE)) {
        return SCPI_RES_ERR;
    }
    duty_cycle_scaled = (input * 0xFFFF) / 100u;
    i2c.i2c_set_open_loop_duty(duty_cycle_scaled);
    return SCPI_RES_OK;
}

scpi_result_t get_duty(scpi_t* context)
{
    int duty_cycle;
    i2c.i2c_word_read_simple(I2C_SPEED_DUTY, &duty_cycle);
    SCPI_ResultInt(context, duty_cycle);
    return SCPI_RES_OK;
}


const scpi_choice_def_t led_selects[] = {
    { "OFF", 0 },
    { "ON", 1 },       
    SCPI_CHOICE_LIST_END
};
static scpi_result_t set_led1(scpi_t * context) {
    int input = 0;
  
    if (!SCPI_ParamChoice(context, led_selects, &input, TRUE))
        return SCPI_RES_ERR;  
        
    led1 = input;
    return SCPI_RES_OK;
}

scpi_result_t get_led1(scpi_t* context)
{
    int led_value = led1;  
    SCPI_ResultText(context, led_selects[led_value].name);
    return SCPI_RES_OK;
}


scpi_result_t read_word(scpi_t* context)
{
    int address = 0;
    int value;
  
    /* read first parameter if present */
    if (!SCPI_ParamInt(context, &address, TRUE)) {
        return SCPI_RES_ERR;
    }
    
    if((address >= 0x1000) && (address < 0x1200)) {
        address = (address - 0x1000) / 2;
        i2c.i2c_word_read_simple(address, &value);
    } else {
        i2c.i2c_word_read_interpreter(address, &value);
    }   
    
    SCPI_ResultIntBase(context, value, 16);
    return SCPI_RES_OK;
}

scpi_result_t write_word(scpi_t* context)
{
    int address = 0;
    int value = 0;
  
    /* read first parameter: address */
    if (!SCPI_ParamInt(context, &address, TRUE)) {
        return SCPI_RES_ERR;
    }
    
    /* read first parameter: address */
    if (!SCPI_ParamInt(context, &value, TRUE)) {
        return SCPI_RES_ERR;
    }
    
    if((address >= 0x1000) && (address < 0x1200)) {
        address = (address - 0x1000) / 2;
        i2c.i2c_word_write_simple(address, value);
    } else {
        i2c.i2c_word_write_interpreter(address, value);
    }   
    return SCPI_RES_OK;
}


/**
 * Reimplement IEEE488.2 *TST?
 *
 * Result should be 0 if everything is ok
 * Result should be 1 if something goes wrong
 *
 * Return SCPI_RES_OK
 */
static scpi_result_t My_CoreTstQ(scpi_t * context) {

    SCPI_ResultInt(context, 0);

    return SCPI_RES_OK;
}

const scpi_command_t scpi_commands[] = {
    /* IEEE Mandated Commands (SCPI std V1999.0 4.1.1) */
    {"*CLS", SCPI_CoreCls, 0},
    {"*ESE", SCPI_CoreEse, 0},
    {"*ESE?", SCPI_CoreEseQ, 0},
    {"*ESR?", SCPI_CoreEsrQ, 0},
    {"*IDN?", SCPI_CoreIdnQ, 0},
    {"*OPC", SCPI_CoreOpc, 0},
    {"*OPC?", SCPI_CoreOpcQ, 0},
    {"*RST", SCPI_CoreRst, 0},
    {"*SRE", SCPI_CoreSre, 0},
    {"*SRE?", SCPI_CoreSreQ, 0},
    {"*STB?", SCPI_CoreStbQ, 0},
    {"*TST?", My_CoreTstQ, 0},
    {"*WAI", SCPI_CoreWai, 0},

    /* Required SCPI commands (SCPI std V1999.0 4.2.1) */
    {"SYSTem:ERRor[:NEXT]?", SCPI_SystemErrorNextQ, 0},
    {"SYSTem:ERRor:COUNt?", SCPI_SystemErrorCountQ, 0},
    {"SYSTem:VERSion?", SCPI_SystemVersionQ, 0},

    //{"STATus:OPERation?", scpi_stub_callback, 0},
    //{"STATus:OPERation:EVENt?", scpi_stub_callback, 0},
    //{"STATus:OPERation:CONDition?", scpi_stub_callback, 0},
    //{"STATus:OPERation:ENABle", scpi_stub_callback, 0},
    //{"STATus:OPERation:ENABle?", scpi_stub_callback, 0},

    {"STATus:QUEStionable[:EVENt]?", SCPI_StatusQuestionableEventQ, 0},
    //{"STATus:QUEStionable:CONDition?", scpi_stub_callback, 0},
    {"STATus:QUEStionable:ENABle", SCPI_StatusQuestionableEnable, 0},
    {"STATus:QUEStionable:ENABle?", SCPI_StatusQuestionableEnableQ, 0},

    {"STATus:PRESet", SCPI_StatusPreset, 0},

    /* DMM */
    {"TARGet:DUTYcycle", set_duty, 0},
    {"TARGet:DUTYcycle?", get_duty, 0},
    {"TARGet:SPEED", set_duty, 0},
    {"TARGet:SPEED?", get_duty, 0},
    {"LED1", set_led1, 0},
    {"LED1?", get_led1, 0},
    {"MEMory:WORD?", read_word, 0},
    {"MEMory:WORD", write_word, 0},
    

    SCPI_CMD_LIST_END
};

scpi_interface_t scpi_interface = {
    /*.error = */ SCPI_Error,
    /*.write = */ SCPI_Write,
    /*.control = */ SCPI_Control,
    /*.flush = */ SCPI_Flush,
    /*.reset = */ SCPI_Reset,
};

char scpi_input_buffer[SCPI_INPUT_BUFFER_LENGTH];
int scpi_error_queue_data[SCPI_ERROR_QUEUE_SIZE];
static scpi_reg_val_t scpi_regs[SCPI_REG_COUNT];

scpi_t scpi_context = {
    cmdlist : scpi_commands,
    buffer : {
        length : SCPI_INPUT_BUFFER_LENGTH,
        position : 0,
        data : scpi_input_buffer
    },
    param_list : { 0 }, 
    interface : &scpi_interface,
    output_count : 0,
    input_count : 0,
    cmd_error : false,
    error_queue : NULL,
    registers : scpi_regs,
    units : scpi_units_def,
    user_context : NULL,
    parser_state : {
        programHeader : { SCPI_TOKEN_UNKNOWN, NULL, 0 },
        programData : { SCPI_TOKEN_UNKNOWN, NULL, 0 },
        numberOfParameters : 0,
        termination : SCPI_MESSAGE_TERMINATION_NONE
    },
    idn : {"semtech", "na-mote", NULL, "01-02"}
};


