/***************************************************************************//**
 * @file gecko.cpp
 * @brief class for creating a gecko-like object and keeps track of its momvement
 *******************************************************************************
 * @section License
 * <b>(C) Copyright 2015 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
 
#include "gecko.h"

/* Number of pixels used in the Gecko head */
#define NPIXHEAD 16
/* Number of pixels used in one Gecko part */
#define NPARTPIXEL 21

/*
 * Pixel map for a gecko part and head
 * y-crd is encoded in the 4 most significant bits, x-crd is encoded in the 4 least significant bits
 * The upper left corner of the part is assigned coordinates (x,y)=(0,0)
 */

const uint8_t head_px_map[NPIXHEAD] = {0x03, 0x12, 0x14, 0x21, 0x25, 0x30, 0x32, 0x34, 0x36, 0x40, 0x46, 0x51, 0x55, 0x62, 0x63, 0x64};
const uint8_t gecko_px_map[NPARTPIXEL] = {0x02, 0x03, 0x04, 0x11, 0x15, 0x20, 0x23, 0x26, 0x30, 0x32, 0x33, 0x34, 0x36, 0x40, 0x43, 0x46, 0x51, 0x55, 0x62, 0x63, 0x64};

Gecko::Gecko():_length(3){
	_head.init(3, 3, UP, head_px_map, NPIXHEAD);
	_last = _length-1;
    for (int i=0;i<_length;i++){
        _position[i] = (_head.getY() + i +1)*BOARD_WIDTH + _head.getX();
    }

}

void Gecko::move(ColorMemLCD &display, Direction dir){
    
    // Remove the last part of the gecko
	this->removePart(display, this->getX(_last)*STEPSIZE + BOARDERWIDTH/2, (this->getY(_last)+TOPEDGE)*STEPSIZE + BOARDERWIDTH/2);

    // Remove head 
    _head.remove(display);

    // Update the part next to the head 
    _head.setDir(dir);
    _position[_last] = _head.getY()*BOARD_WIDTH + _head.getX();

    switch(dir){
        case UP:
            if (_head.getY() == 0)
            {
                _head.setY(BOARD_HEIGHT-1);
            }
            else
            {
            	_head.setY(_head.getY() - 1);
            }
            break;
        case DOWN:
        	if ( _head.getY() == (BOARD_HEIGHT-1))
        	{
        		_head.setY(0);
        	}
        	else
        	{
        		_head.setY(_head.getY() + 1);
        	}
            break; 
        case RIGHT:
        	if (_head.getX() == (BOARD_WIDTH-1))
        	{
        		_head.setX(0);
        	}
        	else
        	{
        		_head.setX(_head.getX() + 1);
        	}
            break;
        case LEFT:
        	if (_head.getX() == 0)
        	{
        		_head.setX(BOARD_WIDTH-1);
        	}
        	else
        	{
        		  _head.setX(_head.getX() - 1);
        	}
            break;
    }
    
    // Draw the parts of the gecko which have been moved
    _head.draw(display);
    this->drawPart(display, this->getX(_last)*STEPSIZE + BOARDERWIDTH/2, (this->getY(_last)+TOPEDGE)*STEPSIZE + BOARDERWIDTH/2);

    // Update the index of the last part
    if (_last == 0){
        _last += _length;
    }
    _last -= 1;
}

void Gecko::draw(ColorMemLCD &display) const{
	_head.draw(display);

	// Loop over indices of all parts of the gecko 
    for (uint8_t i=0;i<_length;i++){
    	this->drawPart(display, this->getX(i)*STEPSIZE + BOARDERWIDTH/2, (this->getY(i)+TOPEDGE)*STEPSIZE + BOARDERWIDTH/2);
    }
} 

bool Gecko::selfCollision() const{
	/* Check if some pixels have the same coordinate */
	for (int i=0;i<_length;i++)
	{
		if ((_head.getX()==this->getX(i)) && (_head.getY()==this->getY(i))){
			return true;
		}
	}
	return false;
}

void Gecko::increaseLength(ColorMemLCD &display, Direction dir)
{
	if ((_length + 1)> MAXLENGTH) return;

	// Store the tail of the gecko
	uint8_t tempPos = _position[_last] ;

	// Move the gecko
	this->move(display, dir);

	uint8_t temp[MAXLENGTH];

	// Copy the front of the gecko
	for (uint8_t i = _last + 1;i<_length;i++)
	{
		temp[i] = _position[i];
	}

	// Insert the "old" tail
	_position[_last + 1] = tempPos;

	// Insert the front of the gecko 
	for (uint8_t i = _last + 1;i<_length;i++)
	{
		_position[i+1] = temp[i];
	}

	// Draw last part of the gecko
	this->drawPart(display, this->getX(_last+1)*STEPSIZE + BOARDERWIDTH/2, (this->getY(_last+1)+TOPEDGE)*STEPSIZE + BOARDERWIDTH/2);


	// Update length and position of last
	_length += 1;
	_last += 1;
}

bool Gecko::headOccupiesTile(uint8_t x_ref, uint8_t y_ref) const{
	return ((x_ref == _head.getX()) && (y_ref == _head.getY()));
}

bool Gecko::occupiesTile(uint8_t x_ref, uint8_t y_ref) const{
	if ( (x_ref == _head.getX()) && (y_ref == _head.getY()) ){
		return true;
	}

	for (uint8_t i=0;i<_length;i++)
	{
			if ((x_ref == this->getX(i)) && (y_ref == this->getY(i)) )
			{
					return true;
			}
	}
	return false;
}

void Gecko::drawPart(ColorMemLCD &display, uint8_t x, uint8_t y) const
{
	  //
	  // x and y denotes the coordinates of the top left corner
	  // Assumes that x and y are given such that the entire part is on the screen
	  
#if(MULTI_UPDATE)
	display.window( BOARDERWIDTH/2, y, LCD_DISP_WIDTH - BOARDERWIDTH/2, STEPSIZE );
#endif

	for (int i=0;i<NPARTPIXEL;i++){
		display.pixel(x +  static_cast<uint8_t>(gecko_px_map[i] & 0xF), y + static_cast<uint8_t>( (gecko_px_map[i] >> 4) & 0xF), FOREGROUND_COLOR);
	}

#if(MULTI_UPDATE)
	display.update();
#endif

}

void Gecko::removePart(ColorMemLCD &display, uint8_t x, uint8_t y) const
{
	 
	 // x and y denotes the coordinates of the top left corner
	 // Assumes that x and y are given such that the  part is on the screen

#if(MULTI_UPDATE)	 
    display.window( BOARDERWIDTH/2, y, LCD_DISP_WIDTH - BOARDERWIDTH/2, STEPSIZE );
#endif

	for (int i=0;i<NPARTPIXEL;i++){
		display.pixel(x + static_cast<uint8_t>(gecko_px_map[i] & 0xF), y + static_cast<uint8_t>( (gecko_px_map[i] >> 4) & 0xF), BACKGROUND_COLOR);
	}

#if(MULTI_UPDATE)
	display.update();
#endif

}

uint8_t Gecko::getX(uint8_t indx) const{
	return _position[indx] - BOARD_WIDTH*(_position[indx]/BOARD_WIDTH);
}

uint8_t Gecko::getY(uint8_t indx) const{
	return _position[indx]/BOARD_WIDTH;
}


