/**
 * @author James Batchelar
 *
 * @section LICENSE
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * PID Controller with Integral Clamping
 * 
 * 
 *     
 
 * @code
#include "mbed.h"
#include "PID.h"

 * @endcode
 */



#ifndef PidController_H
#define PidController_H
#include "mbed.h"

//-- Constants used in system
const int RATE = 20;       //--(ms) Time that Calculate mehtod is being called       
const int AUTOMATIC = 0;    //-- In automatic then PID Controls Output
const int MANUAL = 1;       //-- In Manual then User Controls Output directly

//-- Constructor 

class PidController{
    public:
        // Public Methods
    
        /** Constructor
        *
        */
        PidController(char);
        
        /** Performs the PID Calculation
        * @param SP - Setpoint (target value) units depends on what PID is controlling
        * @param PV - Process Variable (feedback/ measure value) units depends on what PID is controlling
        * @return Returns If Controller is in Automatic then returns PID controlled signal. In manual returns the user SP multipled by scalar.
        */
        float Calculate(float SP, float PV, float ManualMV);
        
        /** Update Internal Settings
        * @param Bias - Added to the PID Calculation
        * @param PropGain - Proportional Gain
        * @param IntGain - Integral Gain
        * @param DiffGain - Differential Gain
        * @param OutputMin - Minimum Limit for the Output (units are same as setpoint)
        * @param OutputMax - Maximum Limit for the Output (units are same as setpoint)
        * @param OutputScale - Multiplier at End of PID loop to convert from engineering units to signal (eg PWM duty cycle)   
        */    
        void UpdateSettings(float Bias, float PropGain, float IntGain, float DiffGain, float OutputMin, float OutputMax);
        
        /** Update Internal Settings
        * @param OutputMin - Minimum Limit for the Output (units are same as setpoint)
        * @param OutputMax - Maximum Limit for the Output (units are same as setpoint)
        */    
        void UpdateSettings(float OutputMin, float OutputMax);
        
        /** Get the controller mode
        * @return 1 = AUTOMATIC, 0 = MANUAL(User has direct contol on output).
        */
        int GetMode(void){return (int)mode;}
        
        /** Set the controller in Manual Mode (PID turned off SP is writted directly to output)
        */
        void SetManual(void){mode = MANUAL;}
        
        /** Set the controller in Automatic Mode (PID turned on)
        */
        void SetAutomatic(void){mode = AUTOMATIC;}
        
        /** Enable Diagnostics for HMI
        */
        void StartDiag(void);
        
        /** Disable Diagnostics for HMI
        */
        void EndDiag(void);
        
        /** Build a string to send back to HMI to Graph PID
        */
        void BuildDiagMessage(float SP, float PV, float PWM, float PropAction, float IntAction, float DifAction);
        
        /** Check to see if the controller is collecting diagnostics data
        * @return true then a diagnostics message is been created.
        */
        bool DiagnosticsEnabled(void){return collectDiagnostics;}
        
        int GetElapsedtime(void){return elapsedTime;}
        
        // Made Public as im lazy
        char diagMsg[65];
        
    private:
        bool mode;
        
        //-- For Diagnostics to GUI
        bool collectDiagnostics;
        int elapsedTime;
        
        //-- For PID Calculations
        char diagChar;
        float bias;
        float error;
        float lastInput;
        float accumError;
        float minLimit, maxLimit;
        float K_p,K_i,K_d;
        float prevError;
    };
    
#endif
    
    
    



