#include <events/mbed_events.h>
#include <mbed.h>

#include "debugger.h"
#include "ADS1232.h"


static EventQueue eventQueue(/* event count */ 10 * EVENTS_EVENT_SIZE);

// pin definitions
// leds
#define LED_GREEN PB_0
#define LED_RED PB_14
#define LED_BLUE PB_7

// led drivers PWM
#define PWM_LED_G1 PC_7
#define PWM_LED_R1 PB_5
#define PWM_LED_B1 PB_3
#define PWM_LED_G2 PE_13
#define PWM_LED_R2 PA_5
#define PWM_LED_B2 PA_6
#define FD_LED PA_4

#define AUDIO_OUT_L PE_11
#define AUDIO_OUT_R PE_9

#define IMU_I2C_SDA PB_11
#define IMU_I2C_SCL PB_10
#define SLAVE_ADDRESS 0xA0 // as defined in ../i2c_test_board/main.cpp

#define LCD_I2C_SDA PD_13
#define LCD_I2C_SCL PD_12

#define JETSON_I2C_SDA PB_9
#define JETSON_I2C_SCL PB_8

#define CONVEYER_DIR PE_15
#define CONVEYER_RUN PE_14

#define BUTTON1 PC_13

#define ONE_WIRE PC_4

#define SAFETY_SPI_MISO PE_5
#define SAFETY_SPI_MOSI PE_6
#define SAFETY_SPI_SCK PE_2
#define SAFETY_SPI_SS PE_4

#define ENABLE_DUMP PG_3

#define JETSON_CAN_TX  PB_13
#define JETSON_CAN_RX  PB_12
#define MOTOR_CAN_TX  PD_1
#define MOTOR_CAN_RX  PD_0

#define SENSE_24V PA_3

// #define ADS_DRDY PE_7
// #define ADS_SCLK PE_8
// #define ADS_PWDN PF_9

#define ADS_DRDY PE_12
#define ADS_SCLK PE_10
#define ADS_PWDN PF_9 // doubles as an enable


// leds
DigitalOut led_green(LED_GREEN, 0);
DigitalOut led_red(LED_RED, 0);
DigitalOut led_blue(LED_BLUE, 0);

// button1
InterruptIn button1(BUTTON1);

// led drivers PWM
PwmOut pwm_led_g1(PWM_LED_G1);
PwmOut pwm_led_r1(PWM_LED_R1);
PwmOut pwm_led_b1(PWM_LED_B1);
PwmOut pwm_led_g2(PWM_LED_G2);
PwmOut pwm_led_r2(PWM_LED_R2);
PwmOut pwm_led_b2(PWM_LED_B2);
DigitalOut fd_led(FD_LED);

// audio out
PwmOut audio_out_l(AUDIO_OUT_L);
PwmOut audio_out_r(AUDIO_OUT_R);

// i2c controllers
I2C imu_i2c(IMU_I2C_SDA, IMU_I2C_SCL);
I2C lcd_i2c(LCD_I2C_SDA, LCD_I2C_SCL);
I2C jetson_i2c(JETSON_I2C_SDA, JETSON_I2C_SCL);

// Conveyer belt
DigitalOut conveyer_dir(CONVEYER_DIR);
DigitalOut conveyer_run(CONVEYER_RUN);

// one wire
DigitalOut one_wire(ONE_WIRE);

// safety spi
SPISlave spiSlave(SAFETY_SPI_MOSI, SAFETY_SPI_MISO, SAFETY_SPI_SCK, SAFETY_SPI_SS);      // mosi, miso, sclk
#define spiSpeed 10000000

// enable dumpy on power supply
DigitalOut enable_dump(ENABLE_DUMP);

CAN can_jetson(JETSON_CAN_RX, JETSON_CAN_TX);
CAN can_motor(MOTOR_CAN_RX, MOTOR_CAN_TX);

AnalogIn sense_24v(SENSE_24V);

ADS1232 ads(ADS_SCLK, ADS_DRDY, ADS_PWDN);

void configSPI()
{
    spiSlave.format(8,3);
    spiSlave.frequency(spiSpeed);
}

void checkSPI()
{
    if (spiSlave.receive()) {
        int valueFromMaster = spiSlave.read();
        
        if (valueFromMaster == 1){
            led_green = !led_green;
        }
        int temp = 1;
        spiSlave.reply(temp);
    }    
}

void testLEDs()
{
    led_red = !led_red;
    led_blue = !led_blue;
}

void setupLedPWM()
{
    pwm_led_g1.period(1. / 1000.); // 1kHz
    pwm_led_r1.period(1. / 1000.); // 1kHz
    pwm_led_b1.period(1. / 1000.); // 1kHz
    pwm_led_g2.period(1. / 1000.); // 1kHz
    pwm_led_r2.period(1. / 1000.); // 1kHz
    pwm_led_b2.period(1. / 1000.); // 1kHz
    fd_led = 1; // enable drivers
}

void testLedPWM()
{
    pwm_led_g1.write(1.0); // max output
    pwm_led_r1.write(1.0);
    pwm_led_b1.write(1.0);
    pwm_led_g2.write(1.0);
    pwm_led_r2.write(1.0);
    pwm_led_b2.write(1.0);
}

void setupAudio()
{
    audio_out_l.period(1. / 400.); // 400 Hz
    audio_out_r.period(1. / 400.); // 400 Hz
}

void testAudio()
{
    audio_out_l.write(0.1);
    audio_out_r.write(0.1);
}

void testIMU()
{
    // Note: PF9 enable pin is accidentally set correct.

    char write[3];
    write[0] = 0x48;
    write[1] = 0x6F;
    write[2] = 0x69;
    imu_i2c.write(SLAVE_ADDRESS, write, 3);

    wait(0.1);
    // write[0] = 0x49;
    // write[1] = 0x6F;
    // write[2] = 0x69;
    // imu_i2c.write(SLAVE_ADDRESS, write, 3);
    char read[6];
    imu_i2c.read(SLAVE_ADDRESS, read, 6);
    imu_i2c.write(SLAVE_ADDRESS, read, 6);
}

void testLCD()
{
    // Note: PF9 enable pin is accidentally set correct.

    char write[3];
    write[0] = 0x48;
    write[1] = 0x6F;
    write[2] = 0x69;
    lcd_i2c.write(SLAVE_ADDRESS, write, 3);

    wait(0.1);
    // write[0] = 0x49;
    // write[1] = 0x6F;
    // write[2] = 0x69;
    // imu_i2c.write(SLAVE_ADDRESS, write, 3);
    char read[6];
    lcd_i2c.read(SLAVE_ADDRESS, read, 6);
    lcd_i2c.write(SLAVE_ADDRESS, read, 6);
}

void test_jetson_i2c()
{
    // Note: PF9 enable pin is accidentally set correct.

    char write[3];
    write[0] = 0x48;
    write[1] = 0x6F;
    write[2] = 0x69;
    jetson_i2c.write(SLAVE_ADDRESS, write, 3);

    wait(0.1);
    char read[6];
    jetson_i2c.read(SLAVE_ADDRESS, read, 6);
    jetson_i2c.write(SLAVE_ADDRESS, read, 6);
}

void testConveyer()
{
    conveyer_dir = !conveyer_dir;
    conveyer_run = !conveyer_run;
}

void testButton1()
{
    led_red = !led_red;
}

void testOneWire()
{
    one_wire = !one_wire;
}

void testEnableDump()
{
    enable_dump = !enable_dump;
}

void testJetsonCan()
{
    int message_id = 1337;
    char data[6] = "Hello";
    can_jetson.write(CANMessage(message_id, data, 6));
}

void testMotorCan()
{
    int message_id = 1337;
    char data[6] = "Hello";
    can_motor.write(CANMessage(message_id, data, 6));
}

void test_sense_24v()
{
    unsigned short value = sense_24v.read_u16();
    LOG("test_sense_24v: %d\r\n", value);
}

void setup_ads1232()
{
    ads.attach();
}

void test_ads1232()
{
    int32_t value = ads.read();
    LOG("test_ads1232: %d\r\n", value);
}

int main()
{
    setupLedPWM();
    setupAudio();
    setup_ads1232();
    configSPI();

    testLedPWM();
    testAudio();
    testJetsonCan();
    testMotorCan();
    
    eventQueue.call_every(1000, testLEDs);
    eventQueue.call_every(1000, testIMU);
    eventQueue.call_every(1000, testConveyer);
    eventQueue.call_every(1000, testOneWire);
    eventQueue.call_every(100, checkSPI);
    eventQueue.call_every(1000, testEnableDump);
    //eventQueue.call_every(1000, test_sense_24v);
    eventQueue.call_every(1000, test_ads1232);
    eventQueue.call_every(1000, testLCD);
    eventQueue.call_every(1000, test_jetson_i2c);

    button1.fall(eventQueue.event(testButton1));

    eventQueue.dispatch_forever();

    return 0;
}