#include <stdio.h>

#include "lorawan/LoRaWANInterface.h"
#include "lorawan/system/lorawan_data_structures.h"
#include "events/EventQueue.h"

#include "trace_helper.h"
#include "lora_radio_helper.h"
#include "SmartFormat.h"

using namespace events;

#define TX_TIMER                        10000                                   // on envoie un message toutes les 10 secondes
#define MAX_NUMBER_OF_EVENTS            10
#define CONFIRMED_MSG_RETRY_COUNTER     3

static EventQueue ev_queue(MAX_NUMBER_OF_EVENTS *EVENTS_EVENT_SIZE);
static void lora_event_handler(lorawan_event_t event);
static LoRaWANInterface lorawan(radio);
static lorawan_app_callbacks_t callbacks;

int main(void)
{
    lorawan_status_t retcode;
    setup_trace();

    if (lorawan.initialize(&ev_queue) != LORAWAN_STATUS_OK) {                   // Initialisation stack lorawan
        printf("\r\n LoRa initialization failed! \r\n");
        return -1;
    }

    callbacks.events = mbed::callback(lora_event_handler);                      // initialisation callback
    lorawan.add_app_callbacks(&callbacks);

    if (lorawan.set_confirmed_msg_retries(CONFIRMED_MSG_RETRY_COUNTER)          // Modification du nombre de renvoie
            != LORAWAN_STATUS_OK) {
        printf("\r\n set_confirmed_msg_retries failed! \r\n\r\n");
        return -1;
    }

    if (lorawan.enable_adaptive_datarate() != LORAWAN_STATUS_OK) {              // adaptative datarate
        printf("\r\n enable_adaptive_datarate failed! \r\n");
        return -1;
    }

    retcode = lorawan.connect();

    if (retcode == LORAWAN_STATUS_OK ||
            retcode == LORAWAN_STATUS_CONNECT_IN_PROGRESS) {
    } else {
        printf("\r\n Connection error, code = %d \r\n", retcode);
        return -1;
    }

    printf("\r\n Connection - In Progress ...\r\n");

    ev_queue.dispatch_forever();                                                // gestion des événements

    return 0;
}

static void send_message()                                                      // Envoie d'un message
{
    int16_t retcode;
    char tosend[50] = "Message LoRaWAN";

    retcode = lorawan.send(MBED_CONF_LORA_APP_PORT, (uint8_t*)tosend, sizeof(tosend),
                           MSG_CONFIRMED_FLAG);
                           
    /*SmartFormat payload(50,3);
    payload.add_Digital_Input(1);
    payload.add_Temperature(25.2);
    payload.add_Illuminance(17.3);

    retcode = lorawan.send(MBED_CONF_LORA_APP_PORT, payload.getBuffer(), payload.getSize(),
                           MSG_CONFIRMED_FLAG);
    */                                                  
    if (retcode == LORAWAN_STATUS_WOULD_BLOCK) {
        printf("Cancel sending \r\n");
         lorawan.cancel_sending();
         NVIC_SystemReset();
    }

    printf("Retcode : %d \r\n",retcode);
    printf("\r\n %d bytes scheduled for transmission \r\n", retcode);
}

static void lora_event_handler(lorawan_event_t event)                           // gestion des événements
{
    switch (event) {
        case CONNECTED:
            printf("\r\n Connection - Successful \r\n");
            ev_queue.call_every(TX_TIMER, send_message);
            break;
        case DISCONNECTED:
            ev_queue.break_dispatch();
            printf("\r\n Disconnected Successfully \r\n");
            break;
        case TX_DONE:
            printf("\r\n Message Sent to Network Server \r\n");
            break;
        case TX_TIMEOUT:
        case TX_ERROR:
        case TX_CRYPTO_ERROR:
        case TX_SCHEDULING_ERROR:
            printf("\r\n Transmission Error - EventCode = %d \r\n", event);
            break;
        case RX_DONE:
            printf("\r\n Received message from Network Server \r\n");
            break;
        case RX_TIMEOUT:
        case RX_ERROR:
            printf("\r\n Error in reception - Code = %d \r\n", event);
            break;
        case JOIN_FAILURE:
            printf("\r\n OTAA Failed - Check Keys \r\n");
            NVIC_SystemReset();
            break;
        case UPLINK_REQUIRED:
            printf("\r\n Uplink required by NS \r\n");
            break;
        default:
            MBED_ASSERT("Unknown Event");
    }
}
