#ifndef XPLANEIO_H_INCLUDED
#define XPLANEIO_H_INCLUDED

#include <queue>
#include "mbed.h"
#include "EthernetNetIf.h"
#include "UDPSocket.h"
#include "XPlaneUdpDATA.h"
#include "XPlaneUdpDecoder.h"
#include "XPlaneUdpEncoder.h"
#include "XPlaneAnalogIn.h"

#define UDP_RECV_BUFFER_SIZE 512
#define UDP_SEND_BUFFER_SIZE 512

/**
 * XPlaneIO is used to establish networked communication with X-Plane.
 * Methods in this class allow the the client to send data to X-Plane as well as
 * read data that has been sent from X-Plane.
 */
 
class XPlaneIO {
public:

    //XPlaneIO();
    ~XPlaneIO();
    
    /**
     * Read configuration file and initialize.
     */
    bool setup(char * configFilename, Ethernet * ethernet);
    
    /**
     * Check if user wants to perform interactive XPlaneIO diagnostics. If so, enter diagnostics mode.
     */
    void diagnostics(Serial & serialInOut);

    /**
     * Include a DATA message, of the given index, in the UDP map -- for receiving from X-Plane.
     * If a message of the given index already exists in the map, then do nothing.
     * Returns true if a new DATA message was created and added to the map.
     */
    bool addDATAToReceive(int msgIndex);

    /**
     * Include a DATA message, of the given index, in the UDP map -- for sending to X-Plane.
     * If a message of the given index already exists in the map, then do nothing.
     * Returns true if a new DATA message was created and added to the map.
     */
    bool addDATAToSend(int msgIndex);

    /**
     * Return a pointer to the DATA message, of the given index, in the UDP receive map (data received from X-Plane).
     * If a message of the given index does not exist in the map, then return NULL.
     */
    XPlaneUdpDATA * getReceiveDATA(int msgIndex) const;

    /**
     * Return a pointer to the DATA message, of the given index, in the UDP send map (data to send to X-Plane).
     * If a message of the given index does not exist in the map, then return NULL.
     */
    XPlaneUdpDATA * getSendDATA(int msgIndex) const;

    /**
     * Return the float data value, of the given DATA message index, at the given float index (0-7), in the UDP receive map (data received from X-Plane).
     * If a message of the given index does not exist in the map, then return -999.999f.
     */
    float getReceiveDATAValue(int msgIndex, int floatIndex) const;

    /**
     * Set the float data value, of the given DATA message index, at the given float index (0-7), in the UDP send map (data to send to X-Plane).
     * If a message of the given index does not exist in the map, then do nothing.
     */
    void setSendDATAValue(int msgIndex, int floatIndex, float f);

    /**
     * Add an X-Plane DataRef to the queue -- to be sent the next time the 
     * sendUdp() ticker function is called.
     */
    void sendDREF(XPlaneUdpDREF & dref);

    /**
     * Start sending UDP data to X-Plane.
     */
    void startSendingUdp();
    
    /**
     * Stop sending UDP data to X-Plane.
     */
    void stopSendingUdp();
    
    /**
     * Utility method to send data via the UDPSocket to the sendHost.
     */
    int udpSocketSend(char * buf, int len);
    
    // Accessors
    bool debug() const;
    bool reverseByteOrder() const;
    XPlaneUdpDATAMap & recvDATAMap();
    XPlaneUdpDATAMap & sendDATAMap();
    
private:

    bool _debug;
    bool _debugVerbose;
    
    //Serial & _serialInOut;
    
    EthernetNetIf * _eth;     // pointer, because don't want instantiated until gather config for it

    // Set up local and remote hosts.
    Host _recvHost;    // mbed local host
    Host _sendHost;    // IP address and port to which UDP will be sent

    // UDP socket to use for both binding to local address (to recv data) and to send data to remote host.
    UDPSocket _udpSocket;

    int _sendInterval;  // milliseconds
    Ticker _sendTicker;
        
    bool _reverseByteOrder;
    
    XPlaneUdpDATAMap _recvDATAMap;   // keyed by DATA message index
    XPlaneUdpDATAMap _sendDATAMap;   // keyed by DATA message index

    queue<XPlaneUdpDREF> _sendDREFQueue;

    char _udpRecvBuffer[UDP_RECV_BUFFER_SIZE];
    XPlaneUdpDecoder _udpDecoder;

    char _udpSendBuffer[UDP_SEND_BUFFER_SIZE];
    XPlaneUdpEncoder _udpEncoder;

    XPlaneAnalogIn * _xpAnalogIn[6];
    int _xpAnalogInCount;
    
    /**
     * Callback handler for when UDP data has been received.
     * Note that this handler is assigned within the setup() method.
     */
    void onUDPSocketEvent(UDPSocketEvent e);

    /**
     * Ticker function for sending UDP data.
     */
    void sendUdp();

    /**
     * Poll analog inputs, and...
     */
    void pollAnalogIn();
};

#endif // XPLANEIO_H_INCLUDED



/* Config file:

debug=Y
debugVerbose=N

# If mbed_ip_address is not specified, then DHCP will be used.
mbed_ip_address=192 168 10 202
mbed_ip_netmask=255 255 255 0
mbed_ip_gateway=192 168 10 99
mbed_ip_dnssrvr=192 168 10 99

# Ethernet link mode: 0 = AutoNegotiate, 1 = HalfDuplex10, 2 = FullDuplex10, 3 = HalfDuplex100, 4 = FullDuplex100
# Note that AutoNegotiate does not work for 10 Mbit hub. e.g. Use 2 (FullDuplex10) for 10 Mbit hub that does not auto-detect speed.
ethernet_link_mode=0

# Local IP port to bind to. X-Plane should be configured to send data to the mbed_ip_address and this port.
recv_port=49000

# IP address and port to which UDP data will be sent.
xplane_ip_address=192 168 10 106
xplane_ip_port=49000

# Interval (milliseconds) for sending UDP data to X-Plane.
send_interval=250

# Depending on X-Plane host computer (e.g Mac), byte order in UDP packets may need to be reversed.
reverse_byte_order=N

# Analog Inputs - for sending an analog value to X-Plane. Up to six are supported (# = 1-6).
# ain_#_pin       : AnalogIn pin number, 15-20 (p15-20)
# ain_#_scale1    : Scale range 1: input_from  input_to  output_from  output_to
#                   Note: A gap between range1's input_to and range2's input_from will be seen as deadband; no data will be sent to X-Plane in that case.
# ain_#_scale2    : Scale range 2: input_from  input_to  output_from  output_to
# ain_#_msg_idx   : X-Plane DATA message: message index
# ain_#_float_idx : X-Plane DATA message: float index (0-7) within the message

# Analog Input #1:
ain_1_pin=20 
ain_1_scale1=0.525 0.610 1.0 0.0
ain_1_scale2=0.615 0.910 0.0 -1.0
ain_1_msg_idx=8
ain_1_float_idx=1

# Analog Input #2:
ain_2_pin=19
ain_2_scale1=0.520 0.605 1.0 0.0
ain_2_scale2=0.615 0.920 0.0 -1.0
ain_2_msg_idx=8
ain_2_float_idx=0

# Analog Input #3:
ain_3_pin=17
ain_3_scale1=0.510 0.615 1.0 0.0
ain_3_scale2=0.620 0.915 0.0 -1.0
ain_3_msg_idx=8
ain_3_float_idx=2


# X-Plane DATA
#  MsgIdx  8 : 0=elev,      1=ailrn,      2=ruddr
#  MsgIdx 13 : 0=elev trim, 1=ailrn trim, 2=ruddr trim, 3=flap handl, 4=flap postn, 5=slat ratio, 6=sbrak handl, 7=sbrak postn
#  MsgIdx 25 : 0=throttle command
#  MsgIdx 26 : 0=throttle actual

*/