#ifndef RANGER_H_INCLUDED
#define RANGER_H_INCLUDED

#include <vector>
#include <stdarg.h>

/**
 * Perform range-type functions on a list of numbers.
 * The numbers must be in ascending order.
 *
 * Example:
 * @code
 *
 * #include "mbed.h"
 * #include "Ranger.h"
 *
 * AnalogIn hatSwitch(p16);
 *
 * int main() {
 * 
 *     Ranger<float> hatSwitchInputRanger(4,  0.59, 0.68, 0.78, 0.92);
 *     int hatSwitchDegreesByRange[5] = {0, -90, 180, 90, 0};
 *
 *     while(1) {
 *         int newRange = hatSwitchInputRanger.range(hatSwitch.read());
 *         if (hatSwitchInputRanger.rangeChanged()) {
 *             int degrees = hatSwitchDegreesByRange[newRange];
 *         }
 *         wait(0.1);
 *     }
 * }
 * @endcode
 */
 
template<typename N>
class Ranger {
public:
    
    /**
     * Create a Ranger object from the given set of passed-in numbers.
     * The numbers in the variable argument list must be in ascending order.
     *
     * @param argCount number of arguments that follow
     * @param ... numbers, in ascending order, which define the ranges 
     */
    Ranger(int argCount, ... );

    /**
     * Create a Ranger object from the given vector.
     * The numbers in the vector must be in ascending order.
     *
     * @param vect vector to copy from
     */
    Ranger(vector<N> vect);

    /**
     * Create a Ranger object from the given array.
     * The numbers in the array must be in ascending order.
     *
     * @param count number of elements to copy from the array
     * @param arr array to copy from
     */
    Ranger(int count, N arr[]);

    /**
     * Return the range that the given number falls in.
     * The range is a slice index (zero-based), based on the Ranger's vector/array.
     *
     * @param input value to determine range from
     */
    int range(N input);

    /**
     * Check if the last call to range() resulted in a change of range from the previous call.
     */
    bool rangeChanged() const;
    
    /**
     * Print a representation of this instance to outputStream.
     */
    void toString(FILE * outputStream);

private:

    /**
     * Common initialization, called from each constructor.
     */
    void init();
    
    vector<N> _nbrVector;
    N _lastInput;
    int _lastRange;
    bool _rangeChanged;
};


template<typename N>
void Ranger<N>::init() {
    _lastRange = -1;
    _rangeChanged = false;
}

template<typename N>
Ranger<N>::Ranger(int argCount, ... ) {
    va_list argList;
    va_start(argList, argCount);
    for (int i = 0; i < argCount; i++) {
        _nbrVector.push_back(va_arg(argList, N));
    }
    va_end(argList);
    
    init();
}

template<typename N>
Ranger<N>::Ranger(vector<N> vect) : _nbrVector(vect) {
    init();
}

template<typename N>
Ranger<N>::Ranger(int count, N arr[]) : _nbrVector(arr, arr + count) {
    init();
}

template<typename N>
int Ranger<N>::range(N input) {

    int newRange = -1;
    
    for (int i=0; i < _nbrVector.size(); i++) {
        if (input < _nbrVector[i]) {
            newRange = i;
            break;
        }
    }
    
    if (newRange == -1) {
        newRange = _nbrVector.size();
    }
    
    _rangeChanged = (_lastRange != newRange);
    _lastInput = input;
    _lastRange = newRange;
    
    return newRange;
}

template<typename N>
bool Ranger<N>::rangeChanged() const {
    return _rangeChanged;
}

template<typename N>
void Ranger<N>::toString(FILE * outputStream) {
    fprintf(outputStream, "Ranger: size=%d, lastRange=%d, changed=%d",
            _nbrVector.size(), _lastRange, _rangeChanged);
}

#endif  // RANGER_H_INCLUDED