#ifndef SCALER_H_INCLUDED
#define SCALER_H_INCLUDED

/**
 * Use linear projection (interpolation) to map/scale values from one range into another.
 *
 * Example:
 * @code
 *
 * #include "mbed.h"
 * #include "Scaler.h"
 *
 * AnalogIn pot(p20);
 *
 * Scaler<float> left(0.965, 0.653, -1, 0);
 * Scaler<float> right(0.653, 0.491, 0, 1);
 *
 * int main() {
 *     while(1) {
 *         float scaledPotValue;
 *         float potValue = pot.read();
 *         printf("Pot value = %f, %d", potValue, pot.read_u16());
 *         if (potValue > 0.653) {
 *             scaledPotValue = left.scale(potValue);
 *         }
 *         else {
 *             scaledPotValue = right.scale(potValue);
 *         }
 *         printf("... scaled: %f \n", scaledPotValue);
 *         wait(0.1);
 *     }
 * }
 * @endcode
 */
 
template<typename N>
class Scaler {
public:
    
    /**
     * Create a Scaler object using the specified input and output ranges.
     *
     * @param inputFrom input range "from"
     * @param inputTo input range "to"
     * @param outputFrom output range "from"
     * @param outputTo output range "to"
     */
    Scaler(N inputFrom, N inputTo, N outputFrom, N outputTo);

    /**
     * Map the input value, within the input range, to the output range.
     *
     * @param input value to project to the output range
     */
    N scale(N input) const;

    // Accessors
    N inputFrom() const;
    N inputTo() const;
    N outputFrom() const;
    N outputTo() const;
    N slope() const;

private:

    N _inputFrom;
    N _inputTo;
    N _outputFrom;
    N _outputTo;
    N _slope;
};


template<typename N>
Scaler<N>::Scaler(N inputFrom, N inputTo, N outputFrom, N outputTo) {
    _inputFrom = inputFrom;
    _inputTo = inputTo;
    _outputFrom = outputFrom;
    _outputTo = outputTo;
    _slope = (outputTo - outputFrom) / (inputTo - inputFrom);
}

template<typename N>
N Scaler<N>::scale(N input) const {
    return _slope * (input - _inputFrom) + _outputFrom;
}

template<typename N>
N Scaler<N>::inputFrom() const {
    return _inputFrom;
}

template<typename N>
N Scaler<N>::inputTo() const {
    return _inputTo;
}

template<typename N>
N Scaler<N>::outputFrom() const {
    return _outputFrom;
}

template<typename N>
N Scaler<N>::outputTo() const {
    return _outputTo;
}

template<typename N>
N Scaler<N>::slope() const {
    return _slope;
}

#endif  // SCALER_H_INCLUDED