/*
 * mbed library for the Sharp memory LCD LS027BDH01 / LS013B4DN04
 * derived from C12832_lcd
 * Copyright (c) 2014 Masato YAMANISHI
 * Released under the MIT License: http://mbed.org/license/mit
 */

/* mbed library for the mbed Lab Board  128*32 pixel LCD
 * use C12832 controller
 * Copyright (c) 2012 Peter Drescher - DC2PD
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SHARP_MLCD_H
#define SHARP_MLCD_H

#include "mbed.h"
#include "GraphicsDisplay.h"

/** optional Defines :
 * #define debug_lcd  1  enable infos to PC_USB
 */
  
#define BPP    1       // Bits per pixel
#define WIDTH  400 // 96
#define HEIGHT 240 // 96
#define PAGES (HEIGHT/8) // 6
#define BUFFER_SIZE (WIDTH*HEIGHT/8)


/** Draw mode
 * NORMAl
 * XOR set pixel by xor the screen
 */
enum {NORMAL,XOR};

/** Bitmap
 */
struct Bitmap{
  int xSize;
  int ySize;
  int Byte_in_Line;
  char* data;
};

class sharp_mlcd : public GraphicsDisplay
{
 public:
  /** Create a aqm1248a object connected to SPI1
   *
   */

  sharp_mlcd(const char* name = "LCD");

  sharp_mlcd(PinName mosi, PinName sclk, PinName scs, PinName extcomin, PinName disp, const char* name = "LCD");

  /** Get the width of the screen in pixel
   *
   * @param
   * @returns width of screen in pixel
   *
   */
  virtual int width();

  /** Get the height of the screen in pixel
   *
   * @returns height of screen in pixel
   *
   */
  virtual int height();

  /** Draw a pixel at x,y black or white
   *
   * @param x horizontal position
   * @param y vertical position
   * @param colour ,1 set pixel ,0 erase pixel
   */
  virtual void pixel(int x, int y,int colour);

  /** draw a circle
   *
   * @param x0,y0 center
   * @param r radius
   * @param colour ,1 set pixel ,0 erase pixel
   *
   */
  void circle(int x, int y, int r, int colour);

  /** draw a filled circle
   *
   * @param x0,y0 center
   * @param r radius
   * @param color ,1 set pixel ,0 erase pixel
   *
   * use circle with different radius,
   * can miss some pixel
   */
  void fillcircle(int x, int y, int r, int colour);

  /** draw a 1 pixel line
   *
   * @param x0,y0 start point
   * @param x1,y1 stop point
   * @param color ,1 set pixel ,0 erase pixel
   *
   */
  void line(int x0, int y0, int x1, int y1, int colour);

  /** draw a rect
   *
   * @param x0,y0 top left corner
   * @param x1,y1 down right corner
   * @param color 1 set pixel ,0 erase pixel
   *                                                   *
   */
  void rect(int x0, int y0, int x1, int y1, int colour);

  /** draw a filled rect
   *
   * @param x0,y0 top left corner
   * @param x1,y1 down right corner
   * @param color 1 set pixel ,0 erase pixel
   *
   */
  void fillrect(int x0, int y0, int x1, int y1, int colour);

  /** copy display buffer to lcd
   *
   */

  void copy_to_lcd(int start = 0);

  /** clear the screen
   *
   */
  virtual void cls(void);

  /** set the drawing mode
   *
   * @param mode NORMAl or XOR
   */

  void setmode(int mode);

  virtual int columns(void);

  /** calculate the max number of columns
   *
   * @returns max column
   * depends on actual font size
   *
   */
  virtual int rows(void);

  /** put a char on the screen
   *
   * @param value char to print
   * @returns printed char
   *
   */
  virtual int _putc(int value);

  /** draw a character on given position out of the active font to the LCD
   *
   * @param x x-position of char (top left)
   * @param y y-position
   * @param c char to print
   *
   */
  virtual void character(int x, int y, int c);

  /** setup cursor position
   *
   * @param x x-position (top left)
   * @param y y-position
   */
  virtual void locate(int x, int y);
    
  /** setup auto update of screen 
   *
   * @param up 1 = on , 0 = off
   * if switched off the program has to call copy_to_lcd() 
   * to update screen from framebuffer
   */
  void set_auto_up(unsigned int up);

  /** get status of the auto update function
   *
   *  @returns if auto update is on
   */
  unsigned int get_auto_up(void);

  /** Vars     */
  SPI _spi;
  // DigitalOut _reset;
  // DigitalOut _A0;
  DigitalOut _EXTCOM;
  DigitalOut _DISP;
  DigitalOut _CS;
  unsigned char* font;
  unsigned int draw_mode;

  // Ticker timer;

  /** select the font to use
   *
   * @param f pointer to font array
   *
   *   font array can created with GLCD Font Creator from http://www.mikroe.com
   *   you have to add 4 parameter at the beginning of the font array to use:
   *   - the number of byte / char
   *   - the vertial size in pixel
   *   - the horizontal size in pixel
   *   - the number of byte per vertical line
   *   you also have to change the array to char[]
   *
   */
  void set_font(unsigned char* f);
    
  /** print bitmap to buffer
   *
   * @param bm Bitmap in flash
   * @param x  x start
   * @param y  y start 
   *
   */

  void print_bm(Bitmap bm, int x, int y, int color);
  void attime();
  void set_reverse_mode(unsigned int r);

 protected:

  /** draw a horizontal line
   *
   * @param x0 horizontal start
   * @param x1 horizontal stop
   * @param y vertical position
   * @param ,1 set pixel ,0 erase pixel
   *
   */
  void hline(int x0, int x1, int y, int colour);

  /** draw a vertical line
   *
   * @param x horizontal position
   * @param y0 vertical start
   * @param y1 vertical stop
   * @param ,1 set pixel ,0 erase pixel
   */
  void vline(int y0, int y1, int x, int colour);

  /** Init the C12832 LCD controller
   *
   */
  void lcd_reset();

  /** Write data to the LCD controller
   *
   * @param dat data written to LCD controller
   *
   */
  void wr_dat(unsigned char value);

  /** Write a command the LCD controller
   *
   * @param cmd: command to be written
   *
   */
  void wr_cmd(unsigned char value);

  void wr_cnt(unsigned char cmd);

  // unsigned int orientation;
  unsigned int char_x;
  unsigned int char_y;
  unsigned char buffer[BUFFER_SIZE];
  unsigned int reverse;
  unsigned int auto_up;

  unsigned int flip;

  unsigned char bit_reverse(unsigned char c);
};

#endif