//-----------------------------------------------------------------------------
// a sample mbed library to play back wave files.
//
// explanation of wave file format.
// https://ccrma.stanford.edu/courses/422/projects/WaveFormat/

// if VERBOSE is uncommented then the wave player will enter a verbose
// mode that displays all data values as it reads them from the file
// and writes them to the DAC.  Very slow and unusable output on the DAC,
// but useful for debugging wave files that don't work.
//#define VERBOSE


#include <mbed.h>
#include <math.h>
#include <stdio.h>
#include <wave_player.h>
#include <shared.h>

Wiistate wiistate = Nothing;
float vol = 0; //VOLUME FLOAT
Mutex key;


//-----------------------------------------------------------------------------
// constructor -- accepts an mbed pin to use for AnalogOut.  Only p18 will work
wave_player::wave_player(AnalogOut *_dac)
{
  wave_DAC=_dac;
  wave_DAC->write_u16(32768);        //DAC is 0-3.3V, so idles at ~1.6V
  verbosity=0;
}

//-----------------------------------------------------------------------------
// if verbosity is set then wave player enters a mode where the wave file
// is decoded and displayed to the screen, including sample values put into
// the DAC FIFO, and values read out of the DAC FIFO by the ISR.  The DAC output
// itself is so slow as to be unusable, but this might be handy for debugging
// wave files that don't play
//-----------------------------------------------------------------------------
void wave_player::set_verbosity(int v)
{
  verbosity=v;
}

int do_vol = 0;
int do_flange = 1;
int do_bit_crush = 0;
int do_delay = 1;

void update_state() {
    Wiistate cur_state;
    key.lock();
        cur_state = wiistate;
    key.unlock();
    
    if(cur_state == Left) {
            do_vol = 1;
            do_flange = 1;
            do_bit_crush = 0;
            do_delay = 0;
    } else if(cur_state == Right) {
            do_vol = 1;
            do_flange = 0;
            do_bit_crush = 1;
            do_delay = 0;
    } else if(cur_state == Up) {
            do_vol = 1;
            do_flange = 0;
            do_bit_crush = 0;
            do_delay = 1;
    } else if(cur_state == Down) {
            do_vol = 1;
            do_flange = 1;
            do_bit_crush = 0;
            do_delay = 1;
    } else if(cur_state == Nothing) {
            do_vol = 0;
            do_flange = 0;
            do_bit_crush = 0;
            do_delay = 0;
            //printf("%d %d %d %d\r\n",  do_vol, do_flange, do_bit_crush, do_delay);
            //printf("%d %d %d %d %d %d\r\n", cur_state==Nothing, Left, Right,Up,Down, Nothing);
    }
    
    //printf("%d %d %d %d\r\n",  do_vol, do_flange, do_bit_crush, do_delay);
}

//-----------------------------------------------------------------------------
// player function.  Takes a pointer to an opened wave file.  The file needs
// to be stored in a filesystem with enough bandwidth to feed the wave data.
// LocalFileSystem isn't, but the SDcard is, at least for 22kHz files.  The
// SDcard filesystem can be hotrodded by increasing the SPI frequency it uses
// internally.
//-----------------------------------------------------------------------------
void wave_player::play(FILE *wavefile)
{
        unsigned chunk_id,chunk_size,channel;
        unsigned data,samp_int,i;
        short unsigned dac_data;
        long long slice_value;
        char *slice_buf;
        short *data_sptr;
        unsigned char *data_bptr;
        int *data_wptr;
        FMT_STRUCT wav_format;
        long slice,num_slices;
  DAC_wptr=0;
  DAC_rptr=0;
  for (i=0;i<256;i+=2) {
    DAC_fifo[i]=0;
    DAC_fifo[i+1]=3000;
  }
  DAC_wptr=4;
  DAC_on=0;

  fread(&chunk_id,4,1,wavefile);
  fread(&chunk_size,4,1,wavefile);


////
#define SAMPLE_RATE 11025
#define D_SIZE 8000
#define M_PI 3.14159265358979323846
short unsigned delayed[D_SIZE];
unsigned int d_rptr = 0;
unsigned int d_wptr = D_SIZE-1000;//((float)D_SIZE)/2.0;
unsigned int d_wptr_2 = D_SIZE-3000;
int flange = 0; // CHANG
int flange_amplitude = D_SIZE/4; // /2
//int flange_freq = SAMPLE_RATE; // 2*pi*cur_samp_cnt/11025
float cur_samp_cnt = 0;
int flange_r_ptr = 0;
int flange_dir = 1;
int flip_sec = 0;

////
for(int di = 0; di < D_SIZE; di++) {
    delayed[di] = 0;
}
  
  while (!feof(wavefile)) {
    if (verbosity)
      printf("Read chunk ID 0x%x, size 0x%x\r\n",chunk_id,chunk_size);
    switch (chunk_id) {
      case 0x46464952:
        fread(&data,4,1,wavefile);
        if (verbosity) {
          printf("RIFF chunk\r\n");
          printf("  chunk size %d (0x%x)\r\n",chunk_size,chunk_size);
          printf("  RIFF type 0x%x\r\n",data);
        }
        break;
      case 0x20746d66:
        fread(&wav_format,sizeof(wav_format),1,wavefile);
        if (verbosity) {
          printf("FORMAT chunk\r\n");
          printf("  chunk size %d (0x%x)\r\n",chunk_size,chunk_size);
          printf("  compression code %d\r\n",wav_format.comp_code);
          printf("  %d channels\r\n",wav_format.num_channels);
          printf("  %d samples/sec\r\n",wav_format.sample_rate);
          printf("  %d bytes/sec\r\n",wav_format.avg_Bps);
          printf("  block align %d\r\n",wav_format.block_align);
          printf("  %d bits per sample\r\n",wav_format.sig_bps);
        }
        if (chunk_size > sizeof(wav_format))
          fseek(wavefile,chunk_size-sizeof(wav_format),SEEK_CUR);
        break;
      case 0x61746164:
// allocate a buffer big enough to hold a slice
        slice_buf=(char *)malloc(wav_format.block_align);
        if (!slice_buf) {
          printf("Unable to malloc slice buffer");
          exit(1);
        }
        num_slices=chunk_size/wav_format.block_align;
        samp_int=1000000/(wav_format.sample_rate);
        if (verbosity) {
          printf("DATA chunk\r\n");
          printf("  chunk size %d (0x%x)\r\n",chunk_size,chunk_size);
          printf("  %d slices\r\n",num_slices);
          printf("  Ideal sample interval=%d\r\n",(unsigned)(1000000.0/wav_format.sample_rate));
          printf("  programmed interrupt tick interval=%d\r\n",samp_int);
        }

// starting up ticker to write samples out -- no printfs until tick.detach is called
        if (verbosity)
          tick.attach_us(this,&wave_player::dac_out, 500000); 
        else
          tick.attach_us(this,&wave_player::dac_out, samp_int); 
        DAC_on=1; 

// start reading slices, which contain one sample each for however many channels
// are in the wave file.  one channel=mono, two channels=stereo, etc.  Since
// mbed only has a single AnalogOut, all of the channels present are averaged
// to produce a single sample value.  This summing and averaging happens in
// a variable of type signed long long, to make sure that the data doesn't
// overflow regardless of sample size (8 bits, 16 bits, 32 bits).
//
// note that from what I can find that 8 bit wave files use unsigned data,
// while 16 and 32 bit wave files use signed data
//


        for (slice=0;slice<num_slices;slice+=1) {
          fread(slice_buf,wav_format.block_align,1,wavefile);
          if (feof(wavefile)) {
            printf("Oops -- not enough slices in the wave file\r\n");
            exit(1);
          }
          data_sptr=(short *)slice_buf;     // 16 bit samples
          data_bptr=(unsigned char *)slice_buf;     // 8 bit samples
          data_wptr=(int *)slice_buf;     // 32 bit samples
          slice_value=0;
          for (channel=0;channel<wav_format.num_channels;channel++) {
            switch (wav_format.sig_bps) {
              case 16:
                if (verbosity)
                  printf("16 bit channel %d data=%d ",channel,data_sptr[channel]);
                slice_value+=data_sptr[channel];
                break;
              case 32:
                if (verbosity)
                  printf("32 bit channel %d data=%d ",channel,data_wptr[channel]);
                slice_value+=data_wptr[channel];
                break;
              case 8:
                if (verbosity)
                  printf("8 bit channel %d data=%d ",channel,(int)data_bptr[channel]);
                slice_value+=data_bptr[channel];
                break;
            }
          }
          slice_value/=wav_format.num_channels;
          
// slice_value is now averaged.  Next it needs to be scaled to an unsigned 16 bit value
// with DC offset so it can be written to the DAC.
          switch (wav_format.sig_bps) {
            case 8:     slice_value<<=8;
                        break;
            case 16:    slice_value+=32768;
                        break;
            case 32:    slice_value>>=16;
                        slice_value+=32768;
                        break;
          }
          dac_data=(short unsigned)slice_value;
          if (verbosity)
            printf("sample %d wptr %d slice_value %d dac_data %u\r\n",slice,DAC_wptr,(int)slice_value,dac_data);
            
            
            
          update_state();
          
            
          ////
          delayed[d_rptr] = dac_data;
          
          d_rptr = (d_rptr == (D_SIZE-1)) ? 0 : (d_rptr + 1);
          d_wptr = (d_wptr == (D_SIZE-1)) ? 0 : (d_wptr + 1);
          d_wptr_2 = (d_wptr_2 == (D_SIZE-1)) ? 0 : (d_wptr_2 + 1);
          
          float f_dac_data = (float)dac_data;
          float f_delay_data;
          float f_delay_data_2;
          int flange_dist = 0;
          int flange_dist_check;
          if(flange) {
            flange_dist = flange_amplitude*sin(2*M_PI*cur_samp_cnt/(float)SAMPLE_RATE);
            /*if(cur_samp_cnt != 0) {
                while(cur_samp_cnt < 4000) {
                    flange_dist = flange_amplitude*sin(2*M_PI*cur_samp_cnt/(float)SAMPLE_RATE);
                printf("%f %f\r\n", flange_dist, 2*M_PI*cur_samp_cnt/(float)SAMPLE_RATE);
                cur_samp_cnt++;
                }
                return;
            }*/
            //printf("%d\r\n",flange_dist);
            flange_dist_check = d_wptr + flange_dist - D_SIZE;
            //d_wptr += flange_dist_check > 0 ? flange_dist_check : (d_wptr + flange_dist) < 0 ? (D_SIZE + d_wptr + flange_dist) : flange_dist;
            //d_wptr += flange_dist_check < 0 ? flange_dist : ((d_wptr + flange_dist) < 0) ? 0 : flange_dist;
            d_wptr = ((d_wptr + flange_dist) > 0) ? (d_wptr + flange_dist) % D_SIZE : (D_SIZE + flange_dist); //(D_SIZE + flange_dist) WRONG
          }
          //cur_samp_cnt = (cur_samp_cnt+1) > SAMPLE_RATE ? 0 : (cur_samp_cnt+1);
          cur_samp_cnt = (cur_samp_cnt+1) > SAMPLE_RATE ? 0 : (cur_samp_cnt+1);
          
          if(cur_samp_cnt == 0) {
              flip_sec = !flip_sec;
          }
          
          
          #define N_SAMPLES_FLANGE_PERIOD 100//((SAMPLE_RATE) >> 2) // WHY IS MZ 100???
          
          if(do_flange) {// && (flip_sec)) {
            if(flange_r_ptr == N_SAMPLES_FLANGE_PERIOD) {
                flange_dir = -1;
            } else if(flange_r_ptr == -N_SAMPLES_FLANGE_PERIOD) {
                flange_dir = 1;   
            }
            flange_r_ptr += flange_dir;   
          } else {
            flange_r_ptr = 0;     
          }
          
          int total_ptr = (d_wptr + flange_r_ptr);
          total_ptr = (total_ptr > D_SIZE) ? (total_ptr - D_SIZE) : total_ptr;
          total_ptr = (total_ptr < 0) ? (D_SIZE - total_ptr) : total_ptr;
          
          //f_delay_data = (float) delayed[d_wptr];
          f_delay_data = (float) delayed[total_ptr];
          f_delay_data_2 = (float) delayed[d_wptr_2];
          float f_mix;
          
          
          if(do_delay || do_flange) {
              if(do_flange) {
                  f_mix = 0.01*f_dac_data + 0.7*f_delay_data;
              } else {
                  f_mix = 0.333*f_dac_data + 0.333*f_delay_data + 0.333*f_delay_data_2;
              }
              //0.01*f_dac_data + 0.7*f_delay_data;
          } else {
              f_mix = f_dac_data;
          }
          ////
          
          float time_in_period = (float)cur_samp_cnt/(float)SAMPLE_RATE;
          //float vol = 0.5 + 0.5*sin(2*M_PI*time_in_period); // Full dynamic range
          float my_vol = 0.75 + 0.25*sin(2*M_PI*time_in_period);
          
          if(do_vol) f_mix = (my_vol*f_mix);
          
          if(do_bit_crush) {
              short unsigned beef = (short unsigned)f_mix;
              //beef = (beef>>7) << 7;
              //printf("beef %x\r\nbeeeef %x\r\n", beef, beef & 0xFFF0);
              DAC_fifo[DAC_wptr]=beef &0xF800;
          } else {
              DAC_fifo[DAC_wptr]=(short unsigned)(f_mix);
          }
          DAC_wptr=(DAC_wptr+1) & 0xff;
          while (DAC_wptr==DAC_rptr) {
          }
        }
        DAC_on=0;
        tick.detach();
        free(slice_buf);
        break;
      case 0x5453494c:
        if (verbosity)
          printf("INFO chunk, size %d\r\n",chunk_size);
        fseek(wavefile,chunk_size,SEEK_CUR);
        break;
      default:
        printf("unknown chunk type 0x%x, size %d\r\n",chunk_id,chunk_size);
        data=fseek(wavefile,chunk_size,SEEK_CUR);
        break;
    }
    fread(&chunk_id,4,1,wavefile);
    fread(&chunk_size,4,1,wavefile);
  }
}


void wave_player::dac_out()
{
  if (DAC_on) {
#ifdef VERBOSE
  printf("ISR rdptr %d got %u\r\n",DAC_rptr,DAC_fifo[DAC_rptr]);
#endif
    wave_DAC->write_u16(DAC_fifo[DAC_rptr]);
    DAC_rptr=(DAC_rptr+1) & 0xff;
  }
}

