/* mbed Microcontroller Library
 * Copyright (c) 2006-2012 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifndef RPC_H
#define RPC_H

#include "mbed.h"
#include "Arguments.h"

namespace mbed {

#define RPC_MAX_STRING      128

struct rpc_function {
    const char *name;
    void (*function_caller)(Arguments*, Reply*);
};

struct rpc_class {
    const char *name;
    const rpc_function *static_functions;
    struct rpc_class *next;
};

/* Class RPC
 *  The RPC class for most things
 */
class RPC {

public:

    RPC(const char *name = NULL);

    virtual ~RPC();

    /* Function get_rpc_methods
     *  Returns a pointer to an array describing the rpc methods
     *  supported by this object, terminated by either
     *  RPC_METHOD_END or RPC_METHOD_SUPER(Superclass).
     *
     * Example
     * > class Example : public RPC {
     * >   int foo(int a, int b) { return a + b; }
     * >   virtual const struct rpc_method *get_rpc_methods() {
     * >     static const rpc_method rpc_methods[] = {
     * >       { "foo", generic_caller<int, Example, int, int, &Example::foo> },
     * >       RPC_METHOD_SUPER(RPC)
     * >     };
     * >     return rpc_methods;
     * >   }
     * > };
     */
    virtual const struct rpc_method *get_rpc_methods();

    static bool call(const char *buf, char *result);

    /* Function lookup
     *  Lookup and return the object that has the given name.
     *
     * Variables
     *  name - the name to lookup.
     */
    static RPC *lookup(const char *name);

protected:
    static RPC *_head;
    RPC *_next;
    char *_name;
    bool _from_construct;

private:
    static rpc_class *_classes;

    static const rpc_function _RPC_funcs[];
    static rpc_class _RPC_class;

    void delete_self();
    static void list_objs(Arguments *args, Reply *result);
    static void clear(Arguments *args, Reply *result);

public:
    /* Function add_rpc_class
     *  Add the class to the list of classes which can have static
     *  methods called via rpc (the static methods which can be called
     *  are defined by that class' get_rpc_class() static method).
     */
    template<class C>
    static void add_rpc_class() {
        rpc_class *c = C::get_rpc_class();
        c->next = _classes;
        _classes = c;
    }

    template<class C>
    static const char *construct() {
        RPC *p = new C();
        p->_from_construct = true;
        return p->_name;
    }
    
    template<class C, typename A1>
    static const char *construct(A1 arg1) {
        RPC *p = new C(arg1);
        p->_from_construct = true;
        return p->_name;
    }
    
    template<class C, typename A1, typename A2>
    static const char *construct(A1 arg1, A2 arg2) {
        RPC *p = new C(arg1, arg2);
        p->_from_construct = true;
        return p->_name;
    }
    
    template<class C, typename A1, typename A2, typename A3>
    static const char *construct(A1 arg1, A2 arg2, A3 arg3) {
        RPC *p = new C(arg1, arg2, arg3);
        p->_from_construct = true;
        return p->_name;
    }
    
    template<class C, typename A1, typename A2, typename A3, typename A4>
    static const char *construct(A1 arg1, A2 arg2, A3 arg3, A4 arg4) {
        RPC *p = new C(arg1, arg2, arg3, arg4);
        p->_from_construct = true;
        return p->_name;
    }
    
    template<class C, typename A1, typename A2, typename A3, typename A4, typename A5>
    static const char *construct(A1 arg1, A2 arg2, A3 arg3, A4 arg4, A5 arg5) {
        RPC *p = new C(arg1, arg2, arg3, arg4, arg5);
        p->_from_construct = true;
        return p->_name;
    }
    
    template<class C, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6>
    static const char *construct(A1 arg1, A2 arg2, A3 arg3, A4 arg4, A5 arg5, A6 arg6) {
        RPC *p = new C(arg1, arg2, arg3, arg4, arg5, arg6);
        p->_from_construct = true;
        return p->_name;
    }
    
    template<class C, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, typename A7>
    static const char *construct(A1 arg1, A2 arg2, A3 arg3, A4 arg4, A5 arg5, A6 arg6, A7 arg7) {
        RPC *p = new C(arg1, arg2, arg3, arg4, arg5, arg6, arg7);
        p->_from_construct = true;
        return p->_name;
    }
    
    template<class C, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, typename A7, typename A8>
    static const char *construct(A1 arg1, A2 arg2, A3 arg3, A4 arg4, A5 arg5, A6 arg6, A7 arg7, A8 arg8) {
        RPC *p = new C(arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
        p->_from_construct = true;
        return p->_name;
    }
};

/* Macro MBED_OBJECT_NAME_MAX
 *  The maximum size of object name (including terminating null byte)
 *  that will be recognised when using fopen to open a FileLike
 *  object, or when using the rpc function.
 */
#define MBED_OBJECT_NAME_MAX 32

/* Macro MBED_METHOD_NAME_MAX
 *  The maximum size of rpc method name (including terminating null
 *  byte) that will be recognised by the rpc function (in rpc.h).
 */
#define MBED_METHOD_NAME_MAX 32

/* Function rpc_method_caller
 */
template<class T, void(T::*member)(const char *, char *)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(arguments, result);
}

/* Function rpc_method_caller
 */
template<class T, void(T::*member)()>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)();
}

/* Function rpc_method_caller
 */
template<class T, typename A1, void(T::*member)(A1)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>()
    );
}

/* Function rpc_method_caller
 */
template<class T, typename A1, typename A2, void(T::*member)(A1, A2)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>()
    );
}

/* Function rpc_method_caller
 */
template<class T, typename A1, typename A2, typename A3, void(T::*member)(A1, A2, A3)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>()
    );
}

/* Function rpc_method_caller
 */
template<class T, typename A1, typename A2, typename A3, typename A4, void(T::*member)(A1, A2, A3, A4)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>()
    );
}

/* Function rpc_method_caller
 */
template<class T, typename A1, typename A2, typename A3, typename A4, typename A5, void(T::*member)(A1, A2, A3, A4, A5)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>()
    );
}

/* Function rpc_method_caller
 */
template<class T, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, void(T::*member)(A1, A2, A3, A4, A5, A6)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>()
    );
}

/* Function rpc_method_caller
 */
template<class T, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, typename A7, void(T::*member)(A1, A2, A3, A4, A5, A6, A7)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>(),
        arguments->getArg<A7>()
    );
}

/* Function rpc_method_caller
 */
template<class T, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, typename A7, typename A8, void(T::*member)(A1, A2, A3, A4, A5, A6, A7, A8)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>(),
        arguments->getArg<A7>(),
        arguments->getArg<A8>()
    );
}
/* Function rpc_method_caller
 */
template<typename R, class T, R(T::*member)()>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)();
    result->putData<R>(res);
}

/* Function rpc_method_caller
 */
template<typename R, class T, typename A1, R(T::*member)(A1)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>()
    );
    result->putData<R>(res);
}

/* Function rpc_method_caller
 */
template<typename R, class T, typename A1, typename A2, R(T::*member)(A1, A2)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>()
    );
    result->putData<R>(res);
}

/* Function rpc_method_caller
 */
template<typename R, class T, typename A1, typename A2, typename A3, R(T::*member)(A1, A2, A3)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>()
    );
    result->putData<R>(res);
}


/* Function rpc_method_caller
 */
template<typename R, class T, typename A1, typename A2, typename A3, typename A4, R(T::*member)(A1, A2, A3, A4)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>()
    );
    result->putData<R>(res);
}


/* Function rpc_method_caller
 */
template<typename R, class T, typename A1, typename A2, typename A3, typename A4, typename A5, R(T::*member)(A1, A2, A3, A4, A5)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>()
    );
    result->putData<R>(res);
}


/* Function rpc_method_caller
 */
template<typename R, class T, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, R(T::*member)(A1, A2, A3, A4, A5, A6)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>()
    );
    result->putData<R>(res);
}

/* Function rpc_method_caller
 */
template<typename R, class T, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, typename A7, R(T::*member)(A1, A2, A3, A4, A5, A6, A7)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>(),
        arguments->getArg<A7>()
    );
    result->putData<R>(res);
}

/* Function rpc_method_caller
 */
template<typename R, class T, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, typename A7, typename A8, R(T::*member)(A1, A2, A3, A4, A5, A6, A7, A8)>
void rpc_method_caller(RPC *this_ptr, Arguments *arguments, Reply *result) {
    R res = (static_cast<T*>(this_ptr)->*member)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>(),
        arguments->getArg<A7>(),
        arguments->getArg<A8>()
    );
    result->putData<R>(res);
}

/* Function rpc_function caller
 */
template<typename R, R(*func)()>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)();
    result->putData<R>(res);
}

/* Function rpc_function caller
 */
template<typename R, typename A1, R(*func)(A1)>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)(
        arguments->getArg<A1>()
    );
    result->putData<R>(res);
}

/* Function rpc_function caller
 */
template<typename R, typename A1, typename A2, R(*func)(A1, A2)>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>()
    );
    result->putData<R>(res);
}

/* Function rpc_function caller
 */
template<typename R, typename A1, typename A2, typename A3, R(*func)(A1, A2, A3)>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>()
    );
    result->putData<R>(res);
}

/* Function rpc_function caller
 */
template<typename R, typename A1, typename A2, typename A3, typename A4, R(*func)(A1, A2, A3, A4)>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>()
    );
    result->putData<R>(res);
}


/* Function rpc_function caller
 */
template<typename R, typename A1, typename A2, typename A3, typename A4, typename A5, R(*func)(A1, A2, A3, A4, A5)>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>()
    );
    result->putData<R>(res);
}


/* Function rpc_function caller
 */
template<typename R, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, R(*func)(A1, A2, A3, A4, A5, A6)>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>()
    );
    result->putData<R>(res);
}


/* Function rpc_function caller
 */
template<typename R, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, typename A7, R(*func)(A1, A2, A3, A4, A5, A6, A7)>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>(),
        arguments->getArg<A7>()
    );
    result->putData<R>(res);
}

/* Function rpc_function caller
 */
template<typename R, typename A1, typename A2, typename A3, typename A4, typename A5, typename A6, typename A7, typename A8, R(*func)(A1, A2, A3, A4, A5, A6, A7, A8)>
void rpc_function_caller(Arguments *arguments, Reply *result) {
    R res = (*func)(
        arguments->getArg<A1>(),
        arguments->getArg<A2>(),
        arguments->getArg<A3>(),
        arguments->getArg<A4>(),
        arguments->getArg<A5>(),
        arguments->getArg<A6>(),
        arguments->getArg<A7>(),
        arguments->getArg<A8>()
    );
    result->putData<R>(res);
}

struct rpc_method {
    const char *name;
    typedef void (*method_caller_t)(RPC*, Arguments*, Reply*);
    typedef const struct rpc_method *(*super_t)(RPC*);
    union {
        method_caller_t method_caller;
        super_t super;
    };
};

template<class C>
const struct rpc_method *rpc_super(RPC *this_ptr) {
    return static_cast<C*>(this_ptr)->C::get_rpc_methods();
}

#define RPC_METHOD_END      { NULL, NULL }
#define RPC_METHOD_SUPER(C) { NULL, (rpc_method::method_caller_t)rpc_super<C> }

} // namespace mbed

#endif