#include "DifferentialDriveKinematics.h"

// TRANSLATION UNIT SCOPE DEFINE
#define PI 3.14159

//DEBUG HELPER
const bool enable_debug = true;
void mydebug(FILE* fp,const char *format , ... ){
   if(enable_debug){
       va_list arglist;
       va_start( arglist, format );
       vfprintf( fp, format, arglist );
       va_end( arglist );
    }
    
}


//CTOR
DifferentialDriveKinematics::DifferentialDriveKinematics(float x, float y, float theta, Motor left_motor, Motor right_motor, HallEffectEncoder& enc_l_, HallEffectEncoder& enc_r_ , ServoLP& penservo_, LSM9DS1& imu_, float dt, float controller_precision):
    _x(x), _y(y),
    _theta(theta),
    Ts(dt),
    m_r(right_motor), m_l(left_motor), 
    enc_l(enc_l_), enc_r(enc_r_),
    penservo(penservo_),
    imu(imu_),
    precision(controller_precision)
{
    // Gains for the control of the angle
    Kp = 0.01;
    Ki = 0.1;
    Kd = 0.1;
    
    // Errors for the control of the angle
    previous_error = 0;
    steady_error = 0;
    
    //BODY
    R = 3.32; // in cm
    L = 16.5; // in cm

    
    // Kalman Matrices Init
    Pk << 1000.0, 0.0, 0.0, 
          0.0, 1000.0, 0.0, 
          0.0, 0.0, 1000.0;

    Hk << 1.0, 0.0, 0.0,
          0.0, 1.0, 0.0,
          0.0, 0.0, 1.0;      
       
    Qk << 25.0, 0.0, 0.0,
          0.0, 25.0, 0.0,
          0.0, 0.0, 25.0;
       

    //Fk not init its a preallocation
          
    Rk_odo << 1.0, 0.0, 0.0,
              0.0, 1.0, 0.0,
              0.0, 0.0, 1.0;
          
    Rk_odo << 10.0,  0.0,  0.0, 
               0.0, 10.0,  0.0,
               0.0,  0.0, 10.0;

}





/*--------------------------------*/
/*        PASSIVE FUNCTION        */
/*--------------------------------*/


void DifferentialDriveKinematics::dynamicsWheels(float speed_left, float speed_right, float previous_states[3], float updated_states[3]) {
    float x = previous_states[0];
    float y = previous_states[1];
    float theta = previous_states[2];
    
    updated_states[0] = x + Ts*(R / 2.0)*((speed_left + speed_right) / R)*cos(theta);
    updated_states[1] = y + Ts*(R / 2.0)*((speed_left + speed_right) / R)*sin(theta);
    
    float pi = PI;
    float temp = theta + Ts*(R / L)*((speed_right - speed_left) / R);
    updated_states[2] = fmod(temp + pi, 2*pi) - pi;
}


void DifferentialDriveKinematics::dynamicsIMU(float linear_speed, float angular_speed, float previous_states[3], float updated_states[3]) {
       float x = previous_states[0];
       float y = previous_states[1];
       float theta = previous_states[2];
       
       updated_states[0] = x + Ts*linear_speed*cos(theta);
       updated_states[1] = y + Ts*linear_speed*sin(theta);
       
       float pi = PI;
       float temp = theta + Ts*angular_speed;
       
       updated_states[2] = fmod(temp + pi, 2*pi) - pi;
     
}



/*
float DifferentialDriveKinematics::getLinearSpeedFromIMU() {
    imu.readAccel();
    return 100*imu.calcAccel(imu.ay)*Ts;
}

float DifferentialDriveKinematics::getAngularSpeedFromIMU() {  
    imu.readGyro();
    
    return  imu.calcGyro(imu.gz); 
}
*/

float DifferentialDriveKinematics::getDifferenceAngle(float angle1, float angle2) {
    float pi = PI;
    float r = fmod(angle2 - angle1, 2*pi);
    if (r < -pi)
        r += 2*pi;
    if (r >= pi)
        r -= 2*pi;
    return r;   
}

float DifferentialDriveKinematics::getAngleFromTarget(float x_target, float y_target) {
    float vecTargetX = x_target - _x;
    float vecTargetY = y_target - _y;
    
//    float vecRobX = cos(_theta);
//    float vecRobY = sin(_theta);
    
//    float dot_product = vecTargetX*vecRobX + vecTargetY*vecRobY;
//    float normTarget = sqrt(vecTargetX*vecTargetX + vecTargetY*vecTargetY);
//    float normRob = sqrt(vecRobX*vecRobX + vecRobY*vecRobY);
    
    return atan2(vecTargetY, vecTargetX);
}



bool DifferentialDriveKinematics::getWheelSpeedFromLinearAngularSpeed(float linear_speed, float angular_speed, float* wheel_speeds) {    
    float Vl = ((2*linear_speed - angular_speed*L) / (2*R))*R;
    float Vr = ((2*linear_speed + angular_speed*L) / (2*R))*R;
 
    // we decided to capp the speed of the wheel to 31 cm/s
    /*if (Vl > 25.0)
        Vl = 25.0;*/
    if (Vl < -25.0)
        Vl = 0;
        
    /*if (Vr > 25.0)
        Vr = 25.0;*/
    if (Vr < -25.0)
        Vr = 0;
    
    
    wheel_speeds[0] = Vl;
    wheel_speeds[1] = Vr;
    
    return true;
}


float DifferentialDriveKinematics::getPWMFromSpeedRightPos(float speed) {
    return (speed - (B_R_Pos)) / A_R_Pos;
}

float DifferentialDriveKinematics::getPWMFromSpeedLeftPos(float speed) {
    return (speed - (B_L_Pos)) / A_L_Pos;
}

float DifferentialDriveKinematics::getPWMFromSpeedRightNeg(float speed) {
    return (speed - (B_R_Neg)) / A_R_Neg;
}

float DifferentialDriveKinematics::getPWMFromSpeedLeftNeg(float speed) {
    return (speed - (B_L_Neg)) / A_L_Neg;
}

float DifferentialDriveKinematics::getPWMFromSpeedRight(float speed) {
    if (speed >= 0) {
        return getPWMFromSpeedRightPos(speed);
    } else {
        return getPWMFromSpeedRightNeg(speed);
    }
}

float DifferentialDriveKinematics::getPWMFromSpeedLeft(float speed) {
    if (speed >= 0) {
        return getPWMFromSpeedLeftPos(speed);
    } else {
        return getPWMFromSpeedLeftNeg(speed);
    }
}




void DifferentialDriveKinematics::kalmanPredict(float speed_left, float speed_right, float predicted_states[3], bool imu = false) {
    
    float previous_states[3];
    previous_states[0] = _x;
    previous_states[1] = _y;
    previous_states[2] = _theta;
    
    dynamicsWheels(speed_left, speed_right, previous_states, predicted_states);
    
    Eigen::Matrix3f& Rk = Rk_odo;
    if(imu){
        Rk = Rk_imu;
    }
    
    Fk << 1, 0, -((speed_left + speed_right)/2)*sin(previous_states[2]),
          0, 1, ((speed_left + speed_right)/2)*cos(previous_states[2]),
          0, 0, 1;
       
    Pk = Fk*Pk*Fk.transpose() + Qk;
    
}


void DifferentialDriveKinematics::kalmanUpdate(float observed_states[3], float predicted_states[3], FILE* fp, bool imu = false) {
    
    float error_x = observed_states[0] - predicted_states[0];
    float error_y = observed_states[1] - predicted_states[1];
    float error_theta = observed_states[2] - predicted_states[2];
    

    Eigen::Matrix3f& Rk = Rk_odo;
    if(imu){
        Rk = Rk_imu;
    }
    
    Eigen::Vector3f yk;
    yk << error_x, error_y, error_theta;
    
    Eigen::Matrix3f Sk = Hk*Pk*Hk.transpose() + Rk;
    
    Eigen::Matrix3f Kk = Pk*Hk.transpose()*Sk.inverse();
    
    Eigen::Vector3f temp = Kk*yk;
    
    _x = predicted_states[0] + temp(0);
    _y = predicted_states[1] + temp(1);
    _theta = predicted_states[2] + temp(2);
    
    Eigen::Matrix3f eye;
    eye << 1, 0, 0,
           0, 1, 0,
           0, 0, 1;
    
    Pk = (eye - Kk*Hk)*Pk;
    
}



/*--------------------------------*/
/*         LOGGER FUNCTION        */
/*--------------------------------*/

void DifferentialDriveKinematics::log(){
    
    // pc->printf("%s",getLog().c_str());
    
}

void DifferentialDriveKinematics::init_logger(Serial* pc_){
    // pc = pc_;
    // log_ticker.attach(this, &DifferentialDriveKinematics::log, 0.1);
    
}

std::string DifferentialDriveKinematics::getLog(float angular_vel, float Vr, float Vl, float real_speed_left, float real_speed_right, float diff_angle, float target_angle) {

    std::ostringstream ss;
    ss << "X: "  << getX();
    ss << " Y: "  << getY();
    ss << " T: "  << getTheta();
    ss << " AV: " << angular_vel;
    ss << " CVr: "<< Vr;
    ss << " CVl: "<< Vl;
    ss << " PMr: "<< getPWMFromSpeedRight(Vr);
    ss << " PMl: "<< getPWMFromSpeedLeft(Vl);
    ss << " RVl: "<< real_speed_left;
    ss << " RVr: "<< real_speed_right;
    ss << " DIFFA: " << diff_angle;
    ss << " TA: " << target_angle;
    ss << " \r\n";
    
    //copy elusion pray
    std::string copyOfStr = ss.str();
   return copyOfStr;
}



/*--------------------------------*/
/*         ACTIVE FUNCTION        */
/*--------------------------------*/
//WIP MOVE SOMEWHERE ELSE
template <typename T> int sgn(T val) {
    return (T(0) < val) - (val < T(0));
}

/* ROBOT STATE TABLE*/
enum Follow_state{
  GOTO_AIM,                       // AIM ANGLE AT TARGET
  GOTO_STOP_TO_RUN,               // STOP before a Run
  GOTO_RUN_UNTIL_CLOSE_OR_MISSED, // RUN UNTIL CLOSE: HIT FINISHED:(GOTO_STOP_OPERATION), HIT(NEXT POINT) OR MISS:(RE AIM SAME POINT)
  
  PEN_OPERATE,                    // Update Pen state, wait a few cycles before proceeding to last_state
  PEN_GET_OFF,
  
  GOTO_STOP_TO_AIM_SAME,          // STOP then got back to GOTO_AIM
  GOTO_STOP_TO_AIM_NEXT,          // STOP then change curr_point and GOTO_AIM
  GOTO_STOP_TO_RETURN             // STOP then return of func

};



void DifferentialDriveKinematics::updateStates(float speed_left, float speed_right) {
    _x = _x + Ts*(R / 2.0)*((speed_left + speed_right) / R)*cos(_theta);
    _y = _y + Ts*(R / 2.0)*((speed_left + speed_right) / R)*sin(_theta);
    
    float pi = PI;
    float temp = _theta + Ts*(R / L)*((speed_right - speed_left) / R);
    _theta = fmod(temp + pi, 2*pi) - pi;
}


bool DifferentialDriveKinematics::followPathV4(Path path, int length_array, Serial* pc, FILE* fp) {
    
     
    //===Internals===
    bool has_finished = false;
    Follow_state state      = PEN_GET_OFF; //Starting state
    
    Follow_state goto_state = GOTO_AIM; //Useful for transient optionnal states
    int waited_cycles = 0;              //Number of cycles to wait in the state
    
    //Wheel dir cached
    float gain_r = 0.0;
    float gain_l = 0.0;
    
    //Speed command for the wheels
    float command_speed_left = 0.0;
    float command_speed_right = 0.0;
    
    //===PATH===
    int curr_id = 0;
    
    float x_curr = 0.0;
    float y_curr = 0.0;
    
    //===GOTO===
    float dist = 10000.0;
    
    //===CONSTANTS===
    const float AIM_angle_margin = 0.05;
    const float AIM_speed = 25;
    const float RUN_dist_margin = 5.0;//5.0
    const float RUN_vel = 23;
    
    const int   PEN_wait_cycles = 10; // Unit in TS, its 0.1 most likely
    const float PEN_UP   = 0.0;
    const float PEN_DOWN = 0.6; //0.6; //WIPWIP DONT WANT TOUCHE THE SOL
    
    mydebug(fp, "GOTO_AIM 0 \r\n");
    
    //===TICK===
    while(not has_finished){

        //===Measurements===
        float real_speed_left  = gain_l*enc_l.getSpeed();
        float real_speed_right = gain_r*enc_r.getSpeed();
        
        //Set Gain 0 if Stopped
        if(real_speed_left  == 0){gain_l = 0.0;}
        if(real_speed_right == 0){gain_r = 0.0;}
        
        //===UPDATE STATE===
        
        updateStates(real_speed_left, real_speed_right);
        
        /*
        float previous_states[3];
        previous_states[0] = _x;
        previous_states[1] = _y;
        previous_states[2] = _theta;
        
        float predicted_states[3];
        predicted_states[0] = 0.0;
        predicted_states[1] = 0.0;
        predicted_states[2] = 0.0;
        
        kalmanPredict(command_speed_left, command_speed_right, predicted_states);
        
        float observed_states[3];
        observed_states[0] = 0.0;
        observed_states[1] = 0.0;
        observed_states[2] = 0.0;
        
        //Update from Wheels Measure
        dynamicsWheels(real_speed_left, real_speed_right, previous_states, observed_states);
        
        //DEBUG
        std::ostringstream ss;
        ss << "observed x: " << observed_states[0];
        ss << " observed y: " << observed_states[1];
        ss << " observed theta: " << observed_states[2];
        ss << " predicted x: " << predicted_states[0];
        ss << " predicted y: " << predicted_states[1];
        ss << " predicted theta: " << predicted_states[2];
        ss << "\r\n";
        fprintf(fp, "%s", ss.str().c_str());
        
        //kalmanUpdate(observed_states, predicted_states, fp );
        */
        
        // IMU UPDATE
        /*
        predicted_states[0] = _x;
        predicted_states[1] = _y;
        predicted_states[2] = _theta;
        
        float linear_speed = getLinearSpeedFromIMU();
        float angular_speed = getAngularSpeedFromIMU();
        
        dynamicsIMU(linear_speed, angular_speed, previous_states, observed_states);

        kalmanUpdate(observed_states, predicted_states, fp,true );
        */

        
        /* ===Assess Stateful Intent=== */

        float wheel_speeds[2] = {0.0,0.0}; //command by default is 0.0 wipwip could create value outside loop
        
        //SET COMMAND
        switch(state){
            
        case GOTO_AIM:
          float target_angle = getAngleFromTarget(x_curr, y_curr);
          float diff_angle   = getDifferenceAngle(_theta, target_angle);
          
          if(abs(diff_angle) > AIM_angle_margin){
            //AIM
            
            int gain_angle= (diff_angle > 0) ? 1.0 : -1.0;
        
            
            wheel_speeds[0] = gain_angle*(-AIM_speed);
            wheel_speeds[1] = gain_angle*(+AIM_speed);
            
            
          }else{
            //Stop a bit
            state = GOTO_STOP_TO_RUN;
            mydebug(fp, "GOTO_STOP_TO_RUN %d \r\n",curr_id);
          }
        
          break;
          
        case GOTO_STOP_TO_RUN:
          //SPEED TO 0
          wheel_speeds[0] = wheel_speeds[1] = 0;
          if(real_speed_left == 0.0 and  real_speed_left == 0.0){
            //Goto same
            waited_cycles = 0; //wipwip
            goto_state =  GOTO_RUN_UNTIL_CLOSE_OR_MISSED; //wipwip
            state = PEN_OPERATE; //GOTO_RUN_UNTIL_CLOSE_OR_MISSED; //GOTO_RUN_UNTIL_CLOSE_OR_MISSED wipwip
            mydebug(fp, "GOTO_RUN_UNTIL_CLOSE_OR_MISSED %d \r\n",curr_id);
            
            //Set Error to 0m needed for run
            dist = 1000000;
          }
          break;
          
        
        case GOTO_RUN_UNTIL_CLOSE_OR_MISSED:
          
          //New dist
          float new_dist = sqrt((_x - x_curr)*(_x - x_curr) + (_y - y_curr)*(_y - y_curr));
          
          if(new_dist <= RUN_dist_margin){
            //HIT
            state = GOTO_STOP_TO_AIM_NEXT;
            mydebug(fp, "Stopping - HIT %d \r\n",curr_id);
            
            
          }else if(new_dist > dist){
            //Missed
            state = GOTO_STOP_TO_AIM_SAME;
            mydebug(fp, "Stopping - MISSED %d \r\n",curr_id);
            
          }else{
            //set command to run
            float angular_vel = 0;
            
            getWheelSpeedFromLinearAngularSpeed(RUN_vel, angular_vel, wheel_speeds);
          }
                
          //WTF ???  
          //float linear_vel = 25; float angular_vel = 0;
          //getWheelSpeedFromLinearAngularSpeed(linear_vel, angular_vel, wheel_speeds); //Give desired wheel speed given linear vel/angle
          
          //update dist
          dist = new_dist;
          break;
          
        case GOTO_STOP_TO_AIM_SAME:
          //SPEED TO 0
          wheel_speeds[0] = wheel_speeds[1] = 0;
          if(real_speed_left == 0.0 and real_speed_left == 0.0){
            //Goto and move pen
            goto_state = GOTO_AIM; 
            waited_cycles = 0;
            state = PEN_GET_OFF; //GOTO_AIM wipwip
            mydebug(fp, "GOTO_AIM %d \r\n", curr_id);
          }
          break;
          
        case GOTO_STOP_TO_AIM_NEXT:
          //SPEED TO 0
          wheel_speeds[0] = wheel_speeds[1] = 0;
          if(real_speed_left == 0.0 and real_speed_left == 0.0){
            //Goto next
            curr_id++;
            
            if(curr_id == length_array){
                state = GOTO_STOP_TO_RETURN;
                mydebug(fp, "GOTO_STOP_TO_RETURN %d \r\n", curr_id); 

            }else{
                x_curr = path[curr_id][0];
                y_curr = path[curr_id][1];
                

                goto_state = GOTO_AIM; 
                waited_cycles = 0;
                state = PEN_GET_OFF; //PEN_OPERATE wipwip
                
            }
            
          }
          
          break;
        
        case PEN_OPERATE:
            if (waited_cycles >= PEN_wait_cycles){
                
                penservo.shutdown();
                state = goto_state;
                break;
            }else if (waited_cycles == 0){
                //START
                if(path[curr_id][2]){
                    penservo.power_and_set(PEN_DOWN);
                }else{
                    penservo.power_and_set(PEN_UP);
                }
                waited_cycles++; //important
            }else{
                //wait
                waited_cycles++;    
            }

            break;
            
        case PEN_GET_OFF:
            if (waited_cycles >= PEN_wait_cycles){
                
                penservo.shutdown();
                state = GOTO_AIM; //must be GOTO AIM
                break;
            }else if (waited_cycles == 0){
                //GET OFF
                penservo.power_and_set(PEN_UP);

                waited_cycles++; //important
            }else{
                //wait
                waited_cycles++;    
            }
        
            break;
          
        case GOTO_STOP_TO_RETURN:
          //SPEED TO 0
          wheel_speeds[0] = wheel_speeds[1] = 0;
          if(real_speed_left == 0.0 and real_speed_left == 0.0){
            //Done
            has_finished = true;
            mydebug(fp, "FINISHED \r\n");
          }
          break;
        
        }
        
        //SET PROPER GAIN
        if(gain_l  == 0.0){gain_l = sgn(wheel_speeds[0]);}
        if(gain_r  == 0.0){gain_r = sgn(wheel_speeds[1]);}
        
        //===Log State=== //before issuing command to tell we just did
        std::string log_string = getLog(0.0, wheel_speeds[1], wheel_speeds[0], real_speed_left, real_speed_right, 0, 0);
        fprintf(fp, "%s", log_string.c_str());
        
        //===Issue Command===
        if( wheel_speeds[0] == 0.0 ){
            m_l.stop(0.75);
        }else{
            m_l.speed(getPWMFromSpeedLeft (wheel_speeds[0])); 
        }
        
        if( wheel_speeds[1] == 0.0 ){
            m_r.stop(0.75);
        }else{
            m_r.speed(getPWMFromSpeedRight (wheel_speeds[1])); 
        }
        
        command_speed_left = wheel_speeds[0];
        command_speed_right = wheel_speeds[1];
        
        //Tickrate
        wait(Ts);
    
    }
    
    
    return true;       

    
}



// FUNCTIONS USING THE IMU

void DifferentialDriveKinematics::updateAngleStateIMU(float angular_vel, float dt) {

    float pi = PI;
    float angle_disp = dt*angular_vel;
    
    float temp = _theta + (angle_disp*PI) / 180.0;
    
    _theta = fmod(temp + pi, 2*pi) - pi;
}

void DifferentialDriveKinematics::updatePositionWheels(float speed_left, float speed_right, float dt) {
    _x = _x + dt*(R / 2.0)*((speed_left + speed_right) / R)*cos(_theta);
    _y = _y + dt*(R / 2.0)*((speed_left + speed_right) / R)*sin(_theta);
}



bool DifferentialDriveKinematics::followPathV5(Path path, int length_array, Serial* pc, FILE* fp) {
       
     //===Internals===
    Follow_state state  = PEN_GET_OFF; //Starting state
    
    Follow_state goto_state = GOTO_AIM; //Useful for transient optionnal states
    int waited_cycles = 0;              //Number of cycles to wait in the state
    
    //Wheel dir cached
    float gain_r = 0.0;
    float gain_l = 0.0;
    
    //Speed command for the wheels
    float command_speed_left = 0.0;
    float command_speed_right = 0.0;

    Timer t;
    
    //===PATH===
    int curr_id = 0;
    
    float x_curr = 0.0;
    float y_curr = 0.0;
    
    //===GOTO===
    float dist = 10000.0;
    
    //===CONSTANTS===
    const float AIM_angle_margin = 0.05;
    const float AIM_speed = 23;
    const float RUN_dist_margin = 5.0;//5.0
    const float RUN_vel = 23;
    
    const int   PEN_wait_cycles = 10; // Unit in TS, its 0.1 most likely
    const float PEN_UP   = 0.0;
    const float PEN_DOWN = 0.6; //0.6; //WIPWIP DONT WANT TOUCHE THE SOL
    
    mydebug(fp, "GOTO_AIM 0 \r\n");
    
    
    for (unsigned int path_index = 0; path_index < length_array; path_index++) {


        float x_curr = path[path_index][0];
        float y_curr = path[path_index][1];

        float pen_state = path[path_index][2];

        float wheel_speeds[2];


        // first go to angle
        float target_angle = getAngleFromTarget(x_curr, y_curr);

        float t1 = 0;
        float t2= 0;
        float ang_vel1 = 0;
        float ang_vel2 = 0;

        float diff_angle = getDifferenceAngle(_theta, target_angle);


        int gain_angle= (diff_angle > 0) ? 1.0 : -1.0;
        
            
        wheel_speeds[0] = gain_angle*(-AIM_speed);
        wheel_speeds[1] = gain_angle*(+AIM_speed);


        m_l.speed(getPWMFromSpeedLeft (wheel_speeds[0]));  

        m_r.speed(getPWMFromSpeedRight (wheel_speeds[1]));
        
        mydebug(fp, "NEW POINT \r\n");
        mydebug(fp, "============== \r\n");
        mydebug(fp, "diff angle: %f \r\n", diff_angle);
        
        t.start();
        while (abs(diff_angle) > 0.05) {
            mydebug(fp, "diff angle: %f \r\n", diff_angle);
            
            // mydebug(fp, "In the aim angle loop \r\n");

            float dt = t2-t1;

            float angular_vel = ((ang_vel1 + ang_vel2)/2.0);
            
            mydebug(fp, "angular vel: %f \r\n", angular_vel);
            
            
            updateAngleStateIMU(angular_vel, dt);

            diff_angle = getDifferenceAngle(_theta, target_angle);



            while(!imu.gyroAvailable());
            imu.readGyro();
            ang_vel1 = ang_vel2;
            ang_vel2 = imu.calcGyro(imu.gz);
            t1 = t2;
            t2 = t.read();

        }
        t.stop();

        m_l.stop(0.75);
        m_r.stop(0.75);
        
        
        
        // then we check if PEN UP or PEN DOWN
        
        //SET PEN
        if(pen_state){
            penservo.power_and_set(PEN_DOWN);
        }else{
            penservo.power_and_set(PEN_UP);
        }
        wait(0.5);
        penservo.shutdown();
        
        


        wait(0.1);

        float dist_to_target = sqrt((_x - x_curr)*(_x - x_curr) + (_y - y_curr)*(_y - y_curr));

        wheel_speeds[0] = RUN_vel;
        wheel_speeds[1] = RUN_vel;

        m_l.speed(getPWMFromSpeedLeft(wheel_speeds[0]));  

        m_r.speed(getPWMFromSpeedRight(wheel_speeds[1]));

        t.start();

        float dt = 0.1;
        bool is_gone_too_far = false;
        float previous_error = 100000;

        // then go to dest
        while(dist_to_target > 3 & !is_gone_too_far) {
            
            mydebug(fp, "dist to target: %f \r\n", dist_to_target);

            wait(dt);

            float real_speed_left  = enc_l.getSpeed();
            float real_speed_right = enc_r.getSpeed();

            updatePositionWheels(real_speed_left, real_speed_right, dt);            
            
            dist_to_target = sqrt((_x - x_curr)*(_x - x_curr) + (_y - y_curr)*(_y - y_curr));
            
            if (dist_to_target > previous_error) {
                mydebug(fp, "HAS GONE TOO FAR, STOP!!! \r\n");
                is_gone_too_far = true;
            }
            previous_error = dist_to_target;
        }
        
        _x = x_curr;
        _y = y_curr;

        m_l.stop(0.75);
        m_r.stop(0.75);
        wait(0.1);
        
        
        // we set the pen to go up after going to target

        penservo.power_and_set(PEN_UP);
        wait(0.5);
        penservo.shutdown();
        
        
        
        // then we go back a little to make sure that the line that are drawn are
        // touching correctly
        
        m_l.speed(getPWMFromSpeedLeft(-23));
        m_r.speed(getPWMFromSpeedRight(-23));
        
        wait(0.6);

        m_l.stop(0.75);
        m_r.stop(0.75);
        
        wait(0.1);


    }


    return true;
    
    
}







float DifferentialDriveKinematics::getX() { return _x; }
float DifferentialDriveKinematics::getY() { return _y; }
float DifferentialDriveKinematics::getTheta() { return _theta; }

float DifferentialDriveKinematics::getR() { return R; }
float DifferentialDriveKinematics::getL() { return L; }