/* Project: Drawing Replication Robot */
/* Arnaud    Billon */
/* Alexandre Mazgaj */

/* INCLUDES */
#include "mbed.h"

//STL
#include<sstream>
#include<string>

// MBDED IO
#include "motordriver.h"
#include <SDFileSystem.h>
#include "LSM9DS1.h"

// Project
#include <HallEffectEncoder.h>
#include <DifferentialDriveKinematics.h>


//==HARDCODED PARAMS==
const float  Ts = 0.05;
float encoder_speed_period = Ts ; //Matched frequencies shown best Performance/Accuracy trade-off

//===IO===
SDFileSystem sd(p5, p6, p7, p8, "sd");

/* ENCODERS */
HallEffectEncoder enc_l(p18, encoder_speed_period); //cant interrupt in p19 p20
HallEffectEncoder enc_r(p17, encoder_speed_period);
/* MOTORS */
Motor m_l(p26, p13, p14, 1); // pwm, fwd1, rev2 // Channel A
Motor m_r(p25, p15, p16, 1); // pwm, fwd1, rev2 // Channel B

/* STATUS LEDS */
DigitalOut led1(LED1); //Standby blink
DigitalOut led3(LED3); //Process Finished
DigitalOut led4(LED4); 
/* DEBUGGING */
Serial  pc(USBTX, USBRX);
/* BUTTON */
InterruptIn toggle_prog_button(p11);
/* PEN SERVO */
//Servo penServo(p23);
ServoLP penServo(p23, p24);




/*=====SYSTEM OPERATION=====*/
volatile bool start = false;

//Watchdog Timer, most conventional way to trigger program reset
class Watchdog {
public:
    void kick(float s) {
        LPC_WDT->WDCLKSEL = 0x1;                // Set CLK src to PCLK
        uint32_t clk = SystemCoreClock / 16;    // WD has a fixed /4 prescaler, PCLK default is /4
        LPC_WDT->WDTC = s * (float)clk;
        LPC_WDT->WDMOD = 0x3;                   // Enabled and Reset
        kick();
    }
    void kick() {
        LPC_WDT->WDFEED = 0xAA;
        LPC_WDT->WDFEED = 0x55;
    }
};
void resetProgram() {
    Watchdog wdt;
    wdt.kick(0.1);
    while (1);       
}
void startProgram() {
    start = true;
}




/* =====CALIBRATION===== */

void calib_speeds(Motor& m_l, Motor& m_r, HallEffectEncoder& enc_l, HallEffectEncoder& enc_r,float min_speed,float max_speed, float num_samples, Serial* pc, FILE* fp){

    pc->printf("=Calibration=\r\n");
    for(float i=min_speed; i<=max_speed; i+= (max_speed - min_speed)/num_samples ){
        //Set Speed
        m_l.speed(i);
        m_r.speed(i);
        //Wait for stability
        wait(0.5);
        //LOG
        std::ostringstream ss;
        ss << "CMD: " << 0.0;
        ss << ", PWM: " << i;
        ss << ", SL: " << enc_l.getSpeed();
        ss << ", SR: " << enc_r.getSpeed();
        ss << "\r\n";
        fprintf(fp, "%s \r\n" ,ss.str().c_str()); 
        // pc->printf("CMD: %f, PWM: %f, SL: %f, SR: %f \r\n", 0.0, i, enc_l.getSpeed(), enc_r.getSpeed());
        //pc->printf("CMD Left: %f, CMD Right: %f, SL: %f, SR: %f\r\n",pwm_left, pwm_right,enc_l.getSpeed(), enc_r.getSpeed());
        wait(0.1);
        
    } 
       
    pc->printf("=Calibration Done=\r\n");
    m_l.speed(0.0f);
    m_r.speed(0.0f);
    
}


/* ===FAST PATH CREATION=== */
void getDebugPath(Path path) {
    
    //SQUARE
    /*
    for (unsigned int i =0; i< 5; i++) {
        path[i][0] = i*10+10;
        path[i][1] = 0;
        path[i][2] = 1;
    }

    for (unsigned int i=0; i < 5; i++){
        path[i+5][0] = 50;
        path[i+5][1] = i*10 +10;  
        path[i+5][2] = 0; 
    }
    
    for (unsigned int i=0; i < 5; i++) {
        path[i+10][0] = 50 -i*10;
        path[i+10][1] = 50;    
        path[i+10][2] = 1;
    }
    
    for (unsigned int i =0; i < 5; i++) {
        path[i+15][0] = 0;
        path[i+15][1] = 50 - i*10;
        path[i+15][2] = 0;
    }
    */
    
    //SQUIGGLY
    path[0 ][0] = 00;path[0 ][1] = 00;path[0 ][2] = 1;
    path[1 ][0] = 20;path[1 ][1] = 00;path[1 ][2] = 1;
    path[2 ][0] = 40;path[2 ][1] = 00;path[2 ][2] = 0;
    path[3 ][0] = 40;path[3 ][1] = 20;path[3 ][2] = 1;
    path[4 ][0] = 20;path[4 ][1] = 20;path[4 ][2] = 1;
    path[5 ][0] = 00;path[5 ][1] = 20;path[5 ][2] = 0;
    path[6 ][0] = 00;path[6 ][1] = 40;path[6 ][2] = 1;
    path[7 ][0] = 20;path[7 ][1] = 40;path[7 ][2] = 1;
    path[8 ][0] = 40;path[8 ][1] = 40;path[8 ][2] = 0;
    path[9 ][0] = 40;path[9 ][1] = 60;path[9 ][2] = 1;
    path[10][0] = 20;path[10][1] = 60;path[10][2] = 1;
    path[11][0] = 00;path[11][1] = 60;path[11][2] = 0;
    
    
}


int path_parser(Path path, FILE* fp) {
    char str[10000];
    int count = 0;
    while (fgets(str, 10000, fp) != NULL) {
        char *used_str = str;
        char *p = str;
        int i = 0;

        while (*p != ';') {
            p++;
            i++;
        }

        char num[20];
        strncpy(num, used_str, i);
        path[count][0] = atof(num);

        ++p;
        used_str = p;
        
        i = 0;
        while (*p != ';') {
            i++;
            p++;
        }

        strncpy(num, used_str, i);
        path[count][1] = atof(num);


        ++p;
        used_str = p;
        i=0;
        while (*p != '\0') {
            i++;
            p++;
        }

        strncpy(num, used_str, i);
        path[count][2] = atof(num);

        count++;


    }
    
    return count;
}




int main() { 
    /* =====INIT===== */
    
    /* SERVO RESET */
    penServo.power_and_set(0.0);
    wait(0.3);
    penServo.shutdown();

    /* SERIAL INIT */
    pc.baud(9600);//115200
    pc.printf("Reset... \r\n");
       
    /* OPEN LOG FILE*/
    FILE *fp = fopen("/sd/logs/path_following_logs.txt", "w");    
    if(fp == NULL) {
        pc.printf("Could not open file for write\n\r");
    }
    fprintf(fp, "Reset... \r\n");
    pc.printf("Opened Log File... \r\n");
    
    /* IMU */ //Forfeited IMU because of bad results
    LSM9DS1 imu(p28, p27, 0xD6, 0x3C);
    imu.begin();
    if (!imu.begin()) {
        pc.printf("Failed to communicate with LSM9DS1.\r\n");
    }
    pc.printf("Calibrating\r\n");
    imu.calibrate(true);
    pc.printf("Done\r\n");
    
    
    /* Dynamics Controller */
    DifferentialDriveKinematics diffDrive(0, 0, 0, m_l, m_r, enc_l,enc_r,penServo, imu, Ts, 0.5);
    
    /* =====STANDBY===== */
    //Wait for button Press
    toggle_prog_button.rise( &startProgram );

    while(!start) {
         led1=1;
         wait(0.5);
         led1=0;
         wait(0.5);   
    }
    wait(1.0);
    
    /* =====START===== */
    led1 = 1;
    //Set reset Capability
    toggle_prog_button.rise( &resetProgram );


    /* GET PATH */
    // Path path ;
    // getDebugPath(path);
    Path path;
    // open file path
    FILE *fp_path = fopen("/sd/paths/drawing.txt", "r");    
    if(fp_path == NULL) {
        pc.printf("Could not open file for write\n\r");
    }
    int n_points = path_parser(path, fp_path);
    fclose(fp_path);    
            
    /* FOLLOW PATH */
    diffDrive.followPathV4(path, n_points, &pc, fp);//num of points
    
    /* =====FINISHED===== */
    led1=0;
    led3=1;
    
    pc.printf("=== OPERATION FINISHED ===\r\n");
    
    //For safety measures
    m_r.speed(0.0f);
    m_l.speed(0.0f);
    wait(1.0);
    
    //Close LOG file
    fclose(fp);
}
